<?php

namespace App\Services;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class DatabaseContext 
{
    public const PREVIEW_USERS_KEY = 'preview_users';

    public static function enableArchiveMode($tempDatabase)
    {
        try {
            $user = app('AppServiceProviderUser');

            // Update user preview status in main database
            DB::connection('mysql_for_check')->table('users')
                ->where('id', $user->id)
                ->update(['preview' => 1]);

            // Store user preview status and temp database name in cache
            $previewUsers = Cache::get(self::PREVIEW_USERS_KEY, []);
            $previewUsers[$user->id] = [
                'start_time' => now(),
                'temp_database' => $tempDatabase
            ];
            Cache::forever(self::PREVIEW_USERS_KEY, $previewUsers);

            Log::info("Archive mode enabled for user: {$user->id} on database: {$tempDatabase}");
        } catch (\Exception $e) {
            Log::error("Error enabling archive mode: " . $e->getMessage());
            throw $e;
        }
    }

    public static function disableArchiveMode()
    {
        try {
            $user = app('AppServiceProviderUser');

            // Update user preview status in main database
            DB::connection('mysql_for_check')->table('users')
                ->where('id', $user->id)
                ->update(['preview' => 0]);

            // Get and remove user from preview cache
            $previewUsers = Cache::get(self::PREVIEW_USERS_KEY, []);
            unset($previewUsers[$user->id]);
            
            if (empty($previewUsers)) {
                Cache::forget(self::PREVIEW_USERS_KEY);
            } else {
                Cache::forever(self::PREVIEW_USERS_KEY, $previewUsers);
            }

            Log::info("Archive mode disabled for user: {$user->id}");
        } catch (\Exception $e) {
            Log::error("Error disabling archive mode: " . $e->getMessage());
            throw $e;
        }
    }

    public static function isArchiveMode($user)
    {
        if (!$user) return false;
        
        $previewUsers = Cache::get(self::PREVIEW_USERS_KEY, []);
        return isset($previewUsers[$user->id]) || $user->preview;
    }

    public static function getCurrentPreviewDatabase($user)
    {
        if (!$user) return null;
        
        $previewUsers = Cache::get(self::PREVIEW_USERS_KEY, []);
        return $previewUsers[$user->id]['temp_database'] ?? null;
    }

    public static function cleanupExpiredPreviews()
    {
        try {
            $previewUsers = Cache::get(self::PREVIEW_USERS_KEY, []);
            $timeout = now()->subHours(24);
            
            foreach ($previewUsers as $userId => $data) {
                if ($data['start_time']->lt($timeout)) {
                    // Drop the temporary database
                    DB::statement("DROP DATABASE IF EXISTS `{$data['temp_database']}`");
                    
                    DB::connection('mysql_for_check')->table('users')
                        ->where('id', $userId)
                        ->update(['preview' => 0]);
                    
                    unset($previewUsers[$userId]);
                    Log::info("Cleaned up expired preview for user: {$userId}");
                }
            }

            if (empty($previewUsers)) {
                Cache::forget(self::PREVIEW_USERS_KEY);
            } else {
                Cache::forever(self::PREVIEW_USERS_KEY, $previewUsers);
            }
        } catch (\Exception $e) {
            Log::error("Error cleaning up previews: " . $e->getMessage());
            throw $e;
        }
    }
}