<?php

namespace App\Http\Controllers\Api\School;

use App\Exports\SchoolEmployeeExport;
use App\Http\Controllers\Controller;
use App\Http\Resources\School\SchoolEmployeeResource;
use App\Http\Resources\School\SchoolEmployeeAttendanceHistoryResource;
use JWTAuth;
use App\Helpers\Helper;

use Illuminate\Http\Request;
use App\Http\Requests\School\SchoolEmployeeRequest;
use App\Models\AdminDashboard\School\School;
use App\Models\AdminDashboard\School\SchoolEmployee;
use App\Models\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Facades\Excel;
use Mail;
use App\Traits\SmsTrait;

class TeacherController extends Controller
{
    use SmsTrait;
    /* to uses the class methods
        -you should be authenticated
        -you should be authorized and have this permission
    */
    public function __construct()
    {
        //$this->middleware(['auth:api','permission:teachers']);
    }


    /** function store school teacher informations */
    public function store(SchoolEmployeeRequest $request){
        $token = $request->bearerToken();
        if($token){
            $user = JWTAuth::toUser($token);
        }
        $school = School::where('user_id',$user->id)->where('id',$request->school_id)->firstOrFail();


        // Validate current_work_id
//        if($request->current_work_id != 2) {
//            return response()->json(['errors' => ['employee-_limit_reached'=>trans('api.current work must be teacher')],
//                'message'=>trans('api.employee_type_not_valid')], 400);
//        }

        $data = $request->all();
        $data['school_id'] = $school->id;
        $data['current_school_year'] = Helper::getSchoolYear();
        $data['type'] = 'teacher';
        $data['have_account'] = 'yes';
        $teacher=SchoolEmployee::create($data);

        /** create users accounts */
        $temp_password =  'teacher'.mt_rand(100000, 999999);
        $this->createUserAccounts($teacher->id,$teacher->name,$teacher->email,$teacher->phone,$temp_password);

        $teacher = new SchoolEmployeeResource($teacher);
        return response()->json(['teacher'=>$teacher,'message'=>trans('api.record stored successfully')]);
    }

    /** function show school teacher informations */
    public function show(Request $request){
        $teacher= SchoolEmployee::findOrFail($request->teacher_id);
        $teacher = new SchoolEmployeeAttendanceHistoryResource($teacher);
        return response()->json(['teacher'=>$teacher,'message'=>trans('api.record details')]);
    }

    /** function update school teacher informations */
    public function update(SchoolEmployeeRequest $request,$id){
        $teacher= SchoolEmployee::findOrFail($id);

        // disapprove change of school manager current work if user update it//
        if ($teacher->current_work_id == 1 && $request->current_work_id != 1) {
            return response()->json(['errors' => ['employee_type_update'=>trans('api.you can not update school manager current work')],
                'message'=>trans('api.employee_type_update')], 400);
        }

        // Consolidate count queries into a single function for maintainability
        $employeeCounts = $this->getEmployeeCounts($teacher->school_id);
        // Validate current_work_id
        if ($teacher->current_work_id != $request->current_work_id && $this->isEmployeeTypeLimitReached($request->current_work_id, $employeeCounts)) {
            return response()->json(['errors' => ['employee-_limit_reached'=>$this->getErrorMessage($request->current_work_id)],
                'message'=>trans('api.employee_limit_reached')], 400);
        }

        $teacher->update($request->except('school_id'));
        $teacher = new SchoolEmployeeResource($teacher);
        return response()->json(['teacher'=>$teacher,'message'=>trans('api.record updated successfully')]);

        return response()->json(['administrative' => $administrative,'message' => trans('api.record updated successfully')]);
    }

    /* this method will delete specific Model recored
        by using ID
    */
    public function destroy(Request $request){

        // Start a database transaction
        DB::beginTransaction();
        try {
            $employee = SchoolEmployee::find($request->teacher_id);
            // Delete related user
            User::where('id', $employee->user_id)->forceDelete();
            // Delete school employees
            $employee->forceDelete();
            // Commit the transaction
            DB::commit();
        } catch (\Exception $e) {
            // Rollback the transaction if something goes wrong
            DB::rollBack();
            // Log the exception or handle the error as needed
            Log::error('Error deleting records: ' . $e->getMessage());
        }
        return response()->json(['message'=>trans('api.record deleted successflly')]);
    }

    /** function create user account for adminstrative */
    public function createUserAccounts($empId,$name,$email,$phone,$temp_password){
        $user = new User();
        $user->name = $name ;
        $user->email = $email ;
        $user->phone = $phone ;
        $user->password = Hash::make($temp_password) ;
        // $user->remember_token = $request->remember_token ;
        $user->status = 'active' ;
        $user->user_type = 'teacher' ;
        $user->verification_code = mt_rand(100000, 999999);
        $user->save();

        /** add user Id to school employee */
        $schoolEmployee = SchoolEmployee::find($empId);
        $schoolEmployee->user_id = $user->id;
        $schoolEmployee->save();

        //send sms & email with system account credentials//
        $this->sendSystemAccount($user->phone, $user->id);
    }

    public function exportTeachers(Request $request){
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');

        $token = $request->bearerToken();
        if($token){
            $user = JWTAuth::toUser($token);
        }

        $school = School::where('user_id',$user->id)->where('id',$request->school_id)->firstOrFail();

        if ($request['date_type'] == "hijri") {
            $startDate = Helper::getCoptic($request['start_date']);
            $endDate = Helper::getCoptic($request['end_date']);
        }
        return Excel::download(new SchoolEmployeeExport($startDate, $endDate,$school->id), 'teachers.xlsx', \Maatwebsite\Excel\Excel::XLSX, [
            'Content-Disposition' => 'attachment; filename="teachers_' . time() . '.xlsx"',
        ]);
    }

    private function getEmployeeCounts($schoolId)
    {
        // Use one query to get all counts
        return [
            'school_manager'=> SchoolEmployee::where('school_id', $schoolId)->where('current_work_id', 1)->count(),
            'educational_administrative' => SchoolEmployee::where('school_id', $schoolId)->where('current_work_id', 6)->count(),
            'students_administrative' => SchoolEmployee::where('school_id', $schoolId)->where('current_work_id', 7)->count(),
            'school_administrative' => SchoolEmployee::where('school_id', $schoolId)->where('current_work_id', 8)->count(),
            'administrative_assistant' => SchoolEmployee::where('school_id', $schoolId)->where('current_work_id', 9)->count(),
            'health_mentor' => SchoolEmployee::where('school_id', $schoolId)->where('current_work_id', 10)->count(),
            'students_mentor' => SchoolEmployee::where('school_id', $schoolId)->where('current_work_id', 11)->count(),
            'learning_resource_creator' => SchoolEmployee::where('school_id', $schoolId)->where('current_work_id', 12)->count(),
            'laboratory_preparator' => SchoolEmployee::where('school_id', $schoolId)->where('current_work_id', 13)->count(),
            'info_recorder_' => SchoolEmployee::where('school_id', $schoolId)->where('current_work_id', 14)->count(),
            'activity_creator' => SchoolEmployee::where('school_id', $schoolId)->where('current_work_id', 15)->count(),
            'school_guard' => SchoolEmployee::where('school_id', $schoolId)->where('current_work_id', 16)->count(),
            'service_worker' => SchoolEmployee::where('school_id', $schoolId)->where('current_work_id', 17)->count(),
            'service_employee' => SchoolEmployee::where('school_id', $schoolId)->where('current_work_id', 18)->count(),
            'school_manager_secretary' => SchoolEmployee::where('school_id', $schoolId)->where('current_work_id', 19)->count(),
        ];
    }

    private function isEmployeeTypeLimitReached($currentWorkId, $employeeCounts)
    {
        // Add conditions for specific work IDs and their limits
        return ($currentWorkId == 1 && $employeeCounts['school_manager']  >= 1) ||
            ($currentWorkId == 6 && $employeeCounts['educational_administrative']  >=1) ||
            ($currentWorkId == 7 && $employeeCounts['students_administrative']  >= 1) ||
            ($currentWorkId == 8 && $employeeCounts['school_administrative']  >= 1) ||
            ($currentWorkId == 9 && $employeeCounts['administrative_assistant'] >= 3)||
            ($currentWorkId == 10 && $employeeCounts['health_mentor'] >= 1)||
            ($currentWorkId == 11 && $employeeCounts['students_mentor'] >= 3)||
            ($currentWorkId == 12 && $employeeCounts['learning_resource_creator'] >= 1)||
            ($currentWorkId == 13 && $employeeCounts['laboratory_preparator'] <= 2)||
            ($currentWorkId == 14 && $employeeCounts['info_recorder_'] >= 2)||
            ($currentWorkId == 15 && $employeeCounts['activity_creator'] >= 1)||
            ($currentWorkId == 16 && $employeeCounts['school_guard'] >= 1)||
            ($currentWorkId == 17 && $employeeCounts['service_worker'] >= 2)||
            ($currentWorkId == 18 && $employeeCounts['service_employee'] >= 1)||
            ($currentWorkId == 19 && $employeeCounts['school_manager_secretary'] >= 1);
    }


    private function getErrorMessage($currentWorkId)
    {
        // Return error messages based on work ID
        switch ($currentWorkId) {
            case 1:
                return trans('api.school_manager_added_before');
            case 6:
                return trans('api.educational_administrative_added_before');
            case 7:
                return trans('api.students_administrative_added_before');
            case 8:
                return trans('api.school_administrative_added_before');
            case 9:
                return trans('api.administrative_assistant_added_before');
            case 10:
                return trans('api.health_mentor_added_before');
            case 11:
                return trans('api.students_mentor_added_before');
            case 12:
                return trans('api.learning_resource_creator_added_before');
            case 13:
                return trans('api.laboratory_preparator_added_before');
            case 14:
                return trans('api.info_recorder_added_before');
            case 15:
                return trans('api.activity_creator_added_before');
            case 16:
                return trans('api.school_guard_added_before');
            case 17:
                return trans('api.service_worker_added_before');
            case 18:
                return trans('api.service_employee_added_before');
            case 19:
                return trans('api.school_manager_secretary_added_before');

            default:
                return trans('api.invalid_work_id');
        }
    }

}
