<?php

namespace App\Http\Controllers\Api\ManagerDashboard\Tables\ClassRoomVisits;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use App\Http\Requests\ManagerDashboard\Tables\ClassRoomVisits\UpdateClassVisitRequest;
use App\Http\Resources\ManagerDashboard\Tables\ClassRoomVisits\ClassroomVisitCollection;
use App\Http\Resources\ManagerDashboard\Tables\ClassRoomVisits\ClassroomVisitResource;
use App\Models\AdminDashboard\School\SchoolSemester;
use App\Models\AdminDashboard\School\School;
use App\Models\ManagerDashboard\Tables\ClassRoomVisits\ClassroomVisit;
use App\Models\Namazeg;
use App\Traits\LogsActivity;
use App\Traits\NamazegTrait;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ClassRoomVisitController extends Controller
{
    use LogsActivity, NamazegTrait;
    /**
     * Display a listing of the resource.
     */
    public function getAll(Request $request, School $school)
    {

        $classRoomVisites =  new ClassroomVisitCollection(SchoolSemester::where('status', '1')->with(['classVisits' => function ($query) use ($school) {
            $query->where('school_id', $school->id); // Filter assignments to the correct school
        }])->paginate($request->per_page));
        return response()->json($classRoomVisites, 200);
    }

    // /**
    //  * Display the specified resource.
    //  */
    public function getSemesterVisits(School $school, SchoolSemester $semester)
    {

        $semesterVisites = ClassroomVisitResource::collection(ClassroomVisit::where('school_id', $school->id)->where('semester_id', $semester->id)->get());
        return response()->json(['semesterVisites' => $semesterVisites], 200);
    }

    /**
     * Update the specified resource in storage.
     */
    public function upsert(UpdateClassVisitRequest $request)
    {
        DB::beginTransaction();

        try {
            $data = $request->validated();
            
            $schoolID = $data['school_id'];

            // Fetch Namazeg records
            $semesterNamazeg = Namazeg::where('form_number', 50)->firstOrFail();
            $namazeg = Namazeg::where('form_number', 229)->firstOrFail();

            // Array to store visits data for bulk insert
            $visits = [];


            // Get all semester IDs for bulk deletion
            $semesterIds = array_column($data['data'], 'semester_id');

            // Delete existing visits in bulk (single query)
            ClassroomVisit::where('school_id', $schoolID)
                ->whereIn('semester_id', $semesterIds)
                ->delete();



            // Process each semester's visits
            foreach ($data['data'] as $semesterVisit) {


                // Create semester namazeg version
                $semesterNamazegVersion = $this->storeNamazegVersion(
                    $semesterNamazeg->id,
                    $semesterNamazeg->internal_extenral,
                    $schoolID
                );

                // Process each visit in the semester
                foreach ($semesterVisit['visites'] as $visit) {
                    // Create namazeg version for the visit
                    $namazegVersion = $this->storeNamazegVersion(
                        $namazeg->id,
                        $namazeg->internal_extenral,
                        $schoolID
                    );

                    // Convert date to Coptic if the date type is Hijri
                    $date = $visit['date'];
                    if ($visit['date_type'] === 'hijri') {
                        $date = Helper::getCoptic($date);
                        $visit['date_type'] = 'coptic';
                    }

                    // Prepare visit data for bulk insert
                    $visits[] = [
                        'school_id' => $schoolID,
                        'semester_namazeg_version_id' => $semesterNamazegVersion->id,
                        'namazeg_version_id' => $namazegVersion->id,
                        'semester_id' => $semesterVisit['semester_id'],
                        'visiting_teacher_id' => $visit['visiting_teacher_id'],
                        'visited_teacher_id' => $visit['visited_teacher_id'],
                        'period_id' => $visit['period_id'],
                        'subject_id' => $visit['subject_id'],
                        'school_class_id' => $visit['school_class_id'],
                        'date_type' => $visit['date_type'],
                        'date' => $date,
                    ];
                }
            }

            // Insert all visits in bulk
            ClassroomVisit::insert($visits);

            // Log activity
            $this->ManagerlogActivity('ClassroomVisit', 'store', $schoolID);

            // Commit the transaction
            DB::commit();

            return response()->json(['message' => trans('api.record updated successfully')], 200);
        } catch (\Exception $e) {
            // Rollback the transaction in case of an error
            DB::rollBack();
          
            // Return an error response
            return response()->json(['message' => 'حدث خطأ، الرجاء المحاولة في وقت لاحق'], 500);
        }
    }
    /**
     * Remove the specified resource from storage.
     */
    public function destroy(ClassroomVisit $classRoomVisit)
    {
        $school_id = 1; //$classRoomVisit->class->school_id;
        $classRoomVisit->delete();
        $this->ManagerlogActivity('ClassroomVisit', 'delete', $school_id);
        return response()->json(['message' => trans('api.record deleted successflly')], 200);
    }
}
