<?php

namespace App\Http\Controllers\Api\ManagerDashboard\StudentAbsenceAndLateness;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use App\Http\Resources\ManagerDashboard\AbsenseAndLateness\StudentLatenessProcedureResource;
use App\Models\AdminDashboard\Procedures\LateProcedure\LateProcedure;
use Illuminate\Http\Request;
use App\Models\AdminDashboard\School\Student;
use App\Models\ManagerDashboard\AbsenseAndLateness\StudentLatenessProcedure;
use App\Models\ManagerDashboard\AbsenseAndLateness\StudentLatenessProcedureMainItem;
use App\Models\ManagerDashboard\AbsenseAndLateness\StudentLatenessProcedureSubItem;
use App\Models\ManagerDashboard\AbsenseAndLateness\StudentLateness;
use App\Models\Namazeg;
use App\Traits\NamazegTrait;
use Illuminate\Support\Facades\DB;

class StudentLatenessProcedureController extends Controller
{
    use NamazegTrait;
    /* to uses the class methods
        -you should be authenticated
        -you should be authorized and have this permission
    */
    public function __construct()
    {
        //$this->middleware(['auth:api','permission:school-student-lateness-procedures']);
    }

    public function getStudentProcedures(Request $request)
    {
        $request->validate([
            'school_id' => 'required|exists:schools,id',
        ]);

        $perPage = $request->per_page ?? 15;
        $semester = Helper::getCurrentSchoolSemester();
        if (!$semester) return response()->json(['error' => trans('api.there is no semester registered for this date')], 404);

        $students = Student::join('students_lateness', 'students_lateness.student_id', '=', 'school_students.id')
            ->join('school_classes', 'students_lateness.class_id', '=', 'school_classes.id')
            ->join('class_names', 'school_classes.class_name_id', '=', 'class_names.id')
            ->select(
                'school_students.id as student_id',
                'school_students.name as student_name',
                'class_names.name as class_name',
                DB::raw('COUNT(students_lateness.student_id) as lateness_days')
            )
            ->where('school_students.school_id', $request->school_id)
            ->where('students_lateness.school_id', $request->school_id)
            ->where('students_lateness.semester_id', $semester->id)
            ->where('status', 'late')
            ->groupBy('school_students.id', 'school_students.name', 'class_names.name')
            ->havingRaw('lateness_days >= 3')
            ->get();

        $students->each(function ($item) use ($request, $semester) {
            $checkIfExists = StudentLatenessProcedure::where('school_id', $request->school_id)
                ->where('student_id', $item->student_id)
                ->where('semester_id', $semester->id)
                ->first();

            $procedure = LateProcedure::where('late_dayes', '>=', $item->lateness_days)->orderBy('late_dayes', 'ASC')->first();
            if (!$procedure) return response()->json(['error' => trans("api.please create procedure for $item->lateness_days lateness days.")], 404);

            if (!$checkIfExists) {
                $general_namozeg = Namazeg::where('form_number', 2)->first();
                $general_namozeg_version_id = $this->storeNamazegVersion(
                    $general_namozeg->id,
                    $general_namozeg->internal_extenral,
                    $request->school_id
                )->id;

                $std_procedure = StudentLatenessProcedure::create([
                    'school_id' => $request->school_id,
                    'semester_id' => $semester->id,
                    'student_id' => $item->student_id,
                    'namozeg_version_id' => $general_namozeg_version_id,
                    'warning' => 'create_procedure',
                    'lateness_days' => $item->lateness_days,
                ]);

                StudentLatenessProcedureMainItem::create([
                    'student_procedure_id' => $std_procedure->id,
                    'procedure_id' => $procedure->id,
                    'status' => 'not_done',
                ]);
            } else {
                $std_procedure = StudentLatenessProcedureMainItem::where('procedure_id', $procedure->id)
                    ->where('student_procedure_id', $checkIfExists->id)
                    ->first();
                if (!$std_procedure) {
                    $checkIfExists->update(['lateness_days' => $procedure->late_dayes]);
                    StudentLatenessProcedureMainItem::create([
                        'student_procedure_id' => $checkIfExists->id,
                        'procedure_id' => $procedure->id,
                        'status' => 'not_done',
                    ]);
                }
            }
        });

        //check for warning
        $studentProcedures = StudentLatenessProcedure::select('id', 'student_id', 'semester_id', 'lateness_days')
            ->where('school_id', $request->school_id)
            ->where('semester_id', $semester->id)
            ->get();

        foreach ($studentProcedures as $studentProcedure) {
            $current = StudentLateness::where('school_id', $request->school_id)
                ->where('semester_id', $studentProcedure->semester_id)
                ->where('student_id', $studentProcedure->student_id)
                ->where('status', 'late')
                ->count();

            if ($current < $studentProcedure->lateness_days) {
                StudentLatenessProcedure::where('school_id', $request->school_id)
                    ->where('semester_id', $studentProcedure->semester_id)
                    ->where('student_id', $studentProcedure->student_id)
                    ->update(['warning' => 'delete_last_procedure']);
            } else {
                $record = StudentLatenessProcedure::with(['mainProcedures' => function ($query) {
                    $query->where('status', 'not_done');
                }])
                    ->where('school_id', $request->school_id)
                    ->where('semester_id', $studentProcedure->semester_id)
                    ->where('student_id', $studentProcedure->student_id)
                    ->first();

                if (count($record->mainProcedures) == 0) {
                    $record->update(['warning' => null]);
                }
            }
        }

        $perPage = $request->per_page ?? 15;

        if ($request->status && $request->status == 'done') {
            $studentProcedures = StudentLatenessProcedure::where('school_id', $request->school_id)
                ->where('semester_id', $semester->id)
                ->whereHas('mainProcedures', function ($query) {
                    $query->where('status', 'done');
                })
                ->paginate($perPage);
        } else {
            $studentProcedures = StudentLatenessProcedure::where('school_id', $request->school_id)
                ->where('semester_id', $semester->id)
                ->paginate($perPage);
        }

        $finalData = StudentLatenessProcedureResource::collection($studentProcedures);

        return response()->json([
            'studentsProcedures' => $finalData,
            'message' => trans(
                'api.list of students available procedures'
            )
        ]);
    }

    public function getProcedureData(LateProcedure $procedure)
    {
        return [
            'procedure_type' => 'lateness',
            'procedure_name' => $procedure->procedureName->name,
            'procedures_to_excute' => $procedure->lateForms()->pluck('name')->toArray(),
            'note' => $procedure->details,
        ];
    }

    public function previewDoneProcedure(StudentLatenessProcedureMainItem $procedure)
    {
        return [
            'procedure_type' => 'lateness',
            'procedure_name' => $procedure->procedure->procedureName->name,
            'procedures_excuted' => $procedure->subProcedures,
        ];
    }

    public function createProcedure(Request $request)
    {
        $request->validate([
            'school_id' => 'required|exists:schools,id',
            'record_id' => 'required|exists:students_lateness_procedures,id',
        ]);

        $studentProcedure = StudentLatenessProcedure::findOrFail($request->record_id);
        $mainProcedure = $studentProcedure->mainProcedures()->where('status', 'not_done')->first();
        $form_numbers = $mainProcedure->procedure->lateForms()->pluck('form_number')->toArray();
        foreach ($form_numbers as $form_number) {
            $namozeg = Namazeg::where('form_number', $form_number)->first();
            $namozeg_version_id = $this->storeNamazegVersion(
                $namozeg->id,
                $namozeg->internal_extenral,
                $request->school_id
            )->id;

            StudentLatenessProcedureSubItem::create([
                'main_item_id' => $mainProcedure->id,
                'procedure_id' => $mainProcedure->procedure->id,
                'namozeg_version_id' => $namozeg_version_id,
            ]);
        }

        $mainProcedure->update(['status' => 'done']);

        return response()->json([
            'message' => trans('api.procedure created successfully')
        ]);
    }

    public function deleteProcedure(StudentLatenessProcedureMainItem $procedure)
    {
        $procedure->delete();
        return response()->json([
            'message' => trans('api.record deleted successfully')
        ]);
    }
}
