<?php

namespace App\Http\Controllers\Api\ManagerDashboard\StudentAbsenceAndLateness;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use App\Http\Resources\ManagerDashboard\AbsenseAndLateness\StudentLatenessResource;
use App\Models\AdminDashboard\School\School;
use App\Traits\SmsTrait;
use Illuminate\Http\Request;
use App\Models\AdminDashboard\School\SchoolClass;
use App\Models\AdminDashboard\School\Student;
use App\Models\ManagerDashboard\AbsenseAndLateness\StudentLateness;
use Tymon\JWTAuth\Facades\JWTAuth;

class StudentLatenessController extends Controller
{
    use SmsTrait;
    /* to uses the class methods
        -you should be authenticated
        -you should be authorized and have this permission
    */
    public function __construct()
    {
        //$this->middleware(['auth:api','permission:school-student-lateness']);
    }

    public function getDayStudents(Request $request)
    {
        // Validate incoming request (optional but recommended)
        $request->validate([
            'school_id' => 'required|exists:schools,id',
            'date_type' => 'nullable|in:coptic,hijri',
        ]);

        if ($request->date) {
            if ($request->date_type == 'hijri') {
                $date = Helper::getCoptic($request->date);
            } else {
                $date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->date)->format('d/m/Y');
            }
        } else {
            $date = date('d/m/Y');
        }

        $semester = Helper::getCurrentSchoolSemester($date);
        if (!$semester) {
            return response()->json(['error' => trans('api.there is no semester registered for this date')], 404);
        }

        $studentIds = Student::where('school_id', $request->school_id);

        if ($request->student_name) {
            $studentIds->where('name', 'LIKE', "{$request->student_name}%");
        }

        if ($request->class_name_id) {
            $class_ids = SchoolClass::where('school_id', $request->school_id)->where('class_name_id', $request->class_name_id)->pluck('id')->toArray();
            $studentIds->whereIn('class_id', $class_ids);
        }

        if ($request->class_number) {
            $class = SchoolClass::where('school_id', $request->school_id)->where('class_name_id', $request->class_name_id)->where('class_number', $request->class_number)->firstOrFail();
            $studentIds->where('class_id', $class->id);
        }

        $studentIds = $studentIds->pluck('id')->toArray();

        $checkIfCreated = StudentLateness::where('date', $date)->where('semester_id', $semester->id)->where('school_id', $request->school_id)->exists();
        if (!$checkIfCreated) {
            foreach ($studentIds as $student_id) {
                $studentLateness = new StudentLateness();
                $studentLateness->school_id = $request->school_id;
                $studentLateness->semester_id = $semester->id;
                $studentLateness->student_id = $student_id;
                $studentLateness->class_id = Student::findOrFail($student_id)?->class_id;
                $studentLateness->status = 'not_late';
                $studentLateness->date = $date;
                $studentLateness->date_type = 'coptic';
                $studentLateness->save();
            }
        }

        // Paginate results with a default value if not provided
        $perPage = $request->per_page ?? 15;

        $studentLateness = StudentLateness::whereIn('student_id', $studentIds)->where('semester_id', $semester->id)->where('school_id', $request->school_id)->where('date', $date)->paginate($perPage);
        $studentLateness = StudentLatenessResource::collection($studentLateness);

        // All Top Statistics
        $schoolStudentsCount = count($studentIds);
        $notLateSchoolStudents = StudentLateness::where('school_id', $request->school_id)->where('semester_id', $semester->id)->where('status', 'not_late')->where('date', $date)->count();
        $lateSchoolStudents = StudentLateness::where('school_id', $request->school_id)
            ->where('date', $date)
            ->where('semester_id', $semester->id)
            ->where(function ($q) {
                $q->where('status', 'late')
                    ->orWhere('status', 'late_with_excuse');
            })->count();
        $StudentsNotRegisteredYet = $schoolStudentsCount - ($notLateSchoolStudents + $lateSchoolStudents);
        $RegisteredClasses = StudentLateness::select('class_id')->where('school_id', $request->school_id)
            ->where('semester_id', $semester->id)
            ->where('date', $date)
            ->whereNotNull('status')
            ->groupBy('class_id')
            ->get();

        $AllClasses = SchoolClass::where('school_id', $request->school_id)->count();

        return response()->json([
            'studentLateness' => $studentLateness,
            'returnStudentsCount' => count($studentLateness),
            'schoolStudentsCount' => $schoolStudentsCount,
            'notLateSchoolStudentsCount' => $notLateSchoolStudents,
            'lateSchoolStudentsCount' => $lateSchoolStudents,
            'StudentsNotRegisteredYetCount' => $StudentsNotRegisteredYet,
            'RegisteredClassesCount' => count($RegisteredClasses),
            'AllClassesCount' => $AllClasses,
            'message' => trans(
                'api.list of students attendance'
            )
        ]);
    }

    /** autocomplete the school Student name */
    public function studentAutocompleteName(Request $request)
    {
        $results = Student::where('school_id', $request->id)->where('name', 'LIKE', "{$request->student_name}%")
            ->select('id', 'name')
            ->get();
        return response()->json($results);
    }

    /** change status of attendance to particular employee */
    public function changeAttendanceStatus(Request $request)
    {
        // Validate incoming request (optional but recommended)
        $rules = [
            'school_id' => 'required|exists:schools,id',
            'date_type' => 'nullable|in:coptic,hijri',
            'records' => 'required|array|min:1', // Ensures records is an array and has at least 1 entry
            'records.*.student_id' => 'required|exists:school_students,id',
            'records.*.status' => 'required|in:not_late,late,late_with_excuse',
        ];

        $request->validate($rules);

        if ($request->date) {
            if ($request->date_type == 'hijri') {
                $date = Helper::getCoptic($request->date);
            } else {
                $date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->date)->format('d/m/Y');
            }
        } else {
            $date = date('d/m/Y');
        }

        $semester = Helper::getCurrentSchoolSemester($date);
        if (!$semester) {
            return response()->json(['error' => trans('api.there is no semester registered for this date')], 404);
        }

        foreach ($request->input('records') as $record) {
            $studentLateness = StudentLateness::where('school_id', $request->school_id)
                ->where('semester_id', $semester->id)
                ->where('student_id', $record['student_id'])
                ->where('date', $date)
                ->firstOrFail();

            if ($record['status'] != "not_late") {
                $user =  JWTAuth::parseToken()->authenticate();
                $message_text = "تم تسجيل تأخر ل " . $studentLateness->student . "اليوم عن الحضور , يرجى متابعة المواظبة.";
                static::sendSMSNotification($studentLateness->student->phone, $message_text, $user->manager_id);
            }

            $studentLateness->update([
                'status' => $record['status'],
                'lateness_reason' => $record['lateness_reason'] ?? null,
            ]);
        }

        return response()->json(['message' => trans('api.student lateness updated successfully')]);
    }
}
