<?php

namespace App\Http\Controllers\Api\ManagerDashboard\StudentAbsenceAndLateness;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use App\Http\Resources\ManagerDashboard\AbsenseAndLateness\StudentAbsenceProcedureResource;
use App\Models\AdminDashboard\Procedures\AbsenceProcedure\AbsenceProcedure;
use Illuminate\Http\Request;
use App\Models\AdminDashboard\School\Student;
use App\Models\ManagerDashboard\AbsenseAndLateness\StudentAbsenceProcedure;
use App\Models\ManagerDashboard\AbsenseAndLateness\StudentAbsenceProcedureMainItem;
use App\Models\ManagerDashboard\AbsenseAndLateness\StudentAbsenceProcedureSubItem;
use App\Models\ManagerDashboard\AbsenseAndLateness\StudentAttendance;
use App\Models\Namazeg;
use App\Traits\NamazegTrait;
use App\Traits\SmsTrait;
use Illuminate\Support\Facades\DB;

class StudentAbsenceProcedureController extends Controller
{
    use NamazegTrait, SmsTrait;
    /* to uses the class methods
        -you should be authenticated
        -you should be authorized and have this permission
    */
    public function __construct()
    {
        //$this->middleware(['auth:api','permission:school-student-attendance-procedures']);
    }

    public function getStudentProcedures(Request $request)
    {
        $request->validate([
            'school_id' => 'required|exists:schools,id',
        ]);

        $semester = Helper::getCurrentSchoolSemester();
        if (!$semester) {
            return response()->json(['error' => trans('api.there is no semester registered for this date')], 404);
        }

        $perPage = $request->per_page ?? 15;

        $students = Student::join('students_attendance', 'students_attendance.student_id', '=', 'school_students.id')
            ->join('school_classes', 'students_attendance.class_id', '=', 'school_classes.id')
            ->join('class_names', 'school_classes.class_name_id', '=', 'class_names.id')
            ->select(
                'school_students.id as student_id',
                'school_students.name as student_name',
                'class_names.name as class_name',
                DB::raw('COUNT(students_attendance.student_id) as absence_days')
            )
            ->where('school_students.school_id', $request->school_id)
            ->where('students_attendance.school_id', $request->school_id)
            ->where('students_attendance.semester_id', $semester->id)
            ->where('students_attendance.status', 'absent')
            ->groupBy('school_students.id', 'school_students.name', 'class_names.name')
            ->havingRaw('absence_days >= 3')
            ->get();

        $absenceProcedures = AbsenceProcedure::orderBy('absence_dayes', 'ASC')->get();
        if ($absenceProcedures->isEmpty()) {
            return response()->json(['error' => trans('api.please create procedures for absence days.')], 404);
        }

        $generalNamozeg = Namazeg::where('form_number', 2)->first();
        if (!$generalNamozeg) {
            return response()->json(['error' => trans('api.general namazeg not found')], 404);
        }

        $students->each(function ($student) use ($request, $semester, $absenceProcedures, $generalNamozeg) {
            $studentProcedure = StudentAbsenceProcedure::firstOrCreate(
                [
                    'school_id' => $request->school_id,
                    'semester_id' => $semester->id,
                    'student_id' => $student->student_id,
                ],
                [
                    'namozeg_version_id' => $this->storeNamazegVersion(
                        $generalNamozeg->id,
                        $generalNamozeg->internal_extenral,
                        $request->school_id
                    )->id,
                    'warning' => 'create_procedure',
                    'absence_days' => $student->absence_days,
                ]
            );

            foreach ($absenceProcedures as $procedure) {
                if ($procedure->absence_dayes <= $student->absence_days) {
                    StudentAbsenceProcedureMainItem::firstOrCreate([
                        'student_procedure_id' => $studentProcedure->id,
                        'procedure_id' => $procedure->id,
                    ], [
                        'status' => 'not_done',
                    ]);
                }
            }
        });

        // Check for warnings
        $this->updateWarnings($request->school_id, $semester->id);

        // Paginate results based on status
        $query = StudentAbsenceProcedure::with('mainProcedures')->where('school_id', $request->school_id)
            ->where('semester_id', $semester->id);

        if ($request->status === 'done') {
            $query->whereHas('mainProcedures', function ($query) {
                $query->where('status', 'done');
            });
        }

        $studentProcedures = $query->paginate($perPage);

        return response()->json([
            'studentsProcedures' => StudentAbsenceProcedureResource::collection($studentProcedures),
            'message' => trans('api.list of students available procedures')
        ]);
    }

    private function updateWarnings($schoolId, $semesterId)
    {
        $studentProcedures = StudentAbsenceProcedure::with(['mainProcedures' => function ($query) {
            $query->where('status', 'not_done');
        }])
            ->where('school_id', $schoolId)
            ->where('semester_id', $semesterId)
            ->get();

        foreach ($studentProcedures as $procedure) {
            $currentAbsences = StudentAttendance::where('school_id', $schoolId)
                ->where('semester_id', $semesterId)
                ->where('student_id', $procedure->student_id)
                ->where('status', 'absent')
                ->count();

            if ($currentAbsences < $procedure->absence_days && $procedure->mainProcedures->isNotEmpty()) {
                $procedure->update(['warning' => 'delete_last_procedure', 'absence_days' => $currentAbsences]);
            } elseif ($procedure->mainProcedures->isEmpty()) {
                $procedure->update(['warning' => null, 'absence_days' => $currentAbsences]);
            }
        }
    }


    public function getProcedureData(AbsenceProcedure $procedure)
    {
        return [
            'procedure_type' => 'absence',
            'procedure_name' => $procedure->procedureName->name,
            'procedures_to_excute' => $procedure->absenceForms()->pluck('name')->toArray(),
            'note' => $procedure->details,
        ];
    }

    public function previewDoneProcedure(StudentAbsenceProcedureMainItem $procedure)
    {
        return [
            'procedure_type' => 'absence',
            'procedure_name' => $procedure->procedure->procedureName->name,
            'procedures_excuted' => $procedure->subProcedures,
        ];
    }

    public function createProcedure(Request $request)
    {
        $request->validate([
            'school_id' => 'required|exists:schools,id',
            'record_id' => 'required|exists:students_absence_procedures,id',
        ]);

        $studentProcedure = StudentAbsenceProcedure::findOrFail($request->record_id);
        $mainProcedure = $studentProcedure->mainProcedures()->where('status', 'not_done')->first();
        $form_numbers = $mainProcedure->procedure->absenceForms()->pluck('form_number')->toArray();

        foreach ($form_numbers as $form_number) {
            $namozeg = Namazeg::where('form_number', $form_number)->first();
            $namozeg_version_id = $this->storeNamazegVersion(
                $namozeg->id,
                $namozeg->internal_extenral,
                $request->school_id
            )->id;

            StudentAbsenceProcedureSubItem::create([
                'main_item_id' => $mainProcedure->id,
                'procedure_id' => $mainProcedure->procedure->id,
                'namozeg_version_id' => $namozeg_version_id,
            ]);
        }

        // $user =  JWTAuth::parseToken()->authenticate();
        // $message_text = "تم اصدار اجراء للطالب" . $studentProcedure->student->name . " لعدم الانتظام فى المواظبة.";
        // static::sendSMSNotification($studentProcedure->student->phone, $message_text, $user->manager_id);

        $mainProcedure->update(['status' => 'done']);

        return response()->json([
            'message' => trans('api.procedure created successfully')
        ]);
    }

    public function deleteProcedure(StudentAbsenceProcedureMainItem $procedure)
    {
        $procedure->delete();
        return response()->json([
            'message' => trans('api.record deleted successfully')
        ]);
    }
}
