<?php

namespace App\Http\Controllers\Api\ManagerDashboard\StudentAbsenceAndLateness;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use App\Http\Resources\ManagerDashboard\AbsenseAndLateness\StudentAttendanceResource;
use App\Models\AdminDashboard\School\SchoolEmployee;
use Illuminate\Http\Request;
use App\Models\AdminDashboard\School\Period;
use App\Models\AdminDashboard\School\SchoolClass;
use App\Models\AdminDashboard\School\Student;
use App\Models\AdminDashboard\School\Subject;
use App\Models\ManagerDashboard\AbsenseAndLateness\PeriodStudentAttendance;
use App\Models\ManagerDashboard\AbsenseAndLateness\StudentAttendance;
use App\Models\ManagerDashboard\Tables\WaitingPeriodsSchedule\WaitingPeriodsSchedule;
use App\Models\ManagerDashboard\TeacherSchedule;
use Illuminate\Support\Facades\DB;

class PeriodStudentAbsenceController extends Controller
{
    /* to uses the class methods
        -you should be authenticated
        -you should be authorized and have this permission
    */
    public function __construct()
    {
        //$this->middleware(['auth:api','permission:school-student-attendance']);
    }

    public function getDatePeriods(Request $request)
    {
        // Validate incoming request (optional but recommended)
        $request->validate([
            'school_id' => 'required|exists:schools,id',
            'period_id' => 'nullable|exists:periods,id',
            'date_type' => 'nullable|in:coptic,hijri',
        ]);

        if ($request->date) {
            if ($request->date_type == "hijri") {
                $date = Helper::getCoptic($request->date);
            } else {
                try {
                    $date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->date)->format('d/m/Y');
                } catch (\Exception $e) {
                    return response()->json(['error' => 'Invalid date format. Please use d/m/Y.'], 400);
                }
            }
        } else {
            $date = date('d/m/Y');
        }

        $semester = Helper::getCurrentSchoolSemester($date);
        if (!$semester) {
            return response()->json(['error' => trans('api.there is no semester registered for this date')], 404);
        }

        $semesterWeekDay = DB::table('semester_week_days')->where('date', $date)->where('semester_id', $semester->id)->first();
        if (!$semesterWeekDay) {
            return response()->json(['error' => trans('api.this day is not a study day')], 404);
        }
        if ($request->period_id) {
            $periods = Period::select('id', 'name')->where('id', $request->period_id)->get();
        } else {
            $periodIds = TeacherSchedule::join('teacher_schedul_data', 'teacher_schedul_data.teacher_schedule_id', 'teacher_schedules.id')
                ->where('school_id', $request->school_id)
                ->where('semester_id', $semester->id)
                // ->where('day_id', $semesterWeekDay->day_id)
                ->distinct('period_id')
                ->orderBy('period_id', 'asc')
                ->pluck('period_id')
                ->toArray();

            $periods = Period::select('id', 'name')->whereIn('id', $periodIds)->get();
        }

        // All Top Statistics
        $schoolStudentsCount = Student::where('school_id', $request->school_id)->count();
        $attendedSchoolStudents = StudentAttendance::where('school_id', $request->school_id)->where('semester_id', $semester->id)->where('status', 'attended')->where('date', $date)->count();
        $absentSchoolStudents = StudentAttendance::where('school_id', $request->school_id)
            ->where('date', $date)
            ->where('semester_id', $semester->id)
            ->where(function ($q) {
                $q->where('status', 'absent')
                    ->orWhere('status', 'absent_with_excuse');
            })->count();

        $finalPeriods = $periods->map(function ($item) use ($request, $date, $semester, $semesterWeekDay, $schoolStudentsCount, $attendedSchoolStudents, $absentSchoolStudents) {

            $perPage = $request->per_page ?? 15;

            $teacherSchedulePeriods = TeacherSchedule::join('teacher_schedul_data', 'teacher_schedul_data.teacher_schedule_id', 'teacher_schedules.id')
                ->select('teacher_schedul_data.id', 'school_employee_id', 'school_class_id', 'subject_id')
                ->where('school_id', $request->school_id)
                ->where('semester_id', $semester->id)
                ->where('period_id', $item->id)
                ->where('day_id', $semesterWeekDay->day_id);


            $waitingPeriodsSchedulePeriods = WaitingPeriodsSchedule::join('waiting_periods_schedule_data', 'waiting_periods_schedule_data.schedule_id', 'waiting_periods_schedules.id')
                ->select('waiting_periods_schedule_data.id', 'waiting_periods_schedule_data.school_employee_id', 'school_class_id', 'subject_id')
                ->where('school_id', $request->school_id)
                ->where('semester_id', $semester->id)
                ->where('day_id', $semesterWeekDay->day_id)
                ->where('creation_date', $date)
                ->where('period_id', $item->id)
                ->where('day_id', $semesterWeekDay->day_id);


            if ($request->teacher_name) {
                $teacherIds = SchoolEmployee::where('school_id', $request->school_id)->where('type', 'teacher')->where('name', 'LIKE', "{$request->teacher_name}%")->pluck('id')->toArray();
                $teacherSchedulePeriods->whereIn('school_employee_id', $teacherIds);
                $waitingPeriodsSchedulePeriods->whereIn('waiting_periods_schedule_data.school_employee_id', $teacherIds);
            }

            if ($request->class_name_id) {
                $classIds = SchoolClass::where('school_id', $request->school_id)->where('class_name_id', $request->class_name_id)->pluck('id')->toArray();
                $teacherSchedulePeriods->whereIn('school_class_id', $classIds);
                $waitingPeriodsSchedulePeriods->whereIn('waiting_periods_schedule_data.school_class_id', $classIds);
            }

            if ($request->class_number) {
                $class = SchoolClass::where('school_id', $request->school_id)->where('class_name_id', $request->class_name_id)->where('class_number', $request->class_number)->firstOrFail();
                $teacherSchedulePeriods->where('school_class_id', $class->id);
                $waitingPeriodsSchedulePeriods->where('waiting_periods_schedule_data.school_class_id', $class->id);
            }

            $teacherSchedulePeriods = $teacherSchedulePeriods->paginate($perPage);
            $waitingPeriodsSchedulePeriods = $waitingPeriodsSchedulePeriods->paginate($perPage);

            $periodDetails = $teacherSchedulePeriods->map(function ($item2) use ($waitingPeriodsSchedulePeriods, $schoolStudentsCount, $attendedSchoolStudents, $absentSchoolStudents, $date, $item, $request, $semester) {
                $typeMatched = $waitingPeriodsSchedulePeriods->where('school_class_id', $item2['school_class_id'])
                    ->where('subject_id', $item2['subject_id'])->first();

                if ($typeMatched) {
                    $currentEmployee = $item2->school_employee_id;
                    $employee_name = SchoolEmployee::findOrFail($currentEmployee)->name;

                    $alternativeEmployee = $typeMatched->school_employee_id;
                    $alternative_employee_name = SchoolEmployee::findOrFail($alternativeEmployee)->name;

                    $item2 = $typeMatched;

                    $item2->school_employee_id = $currentEmployee;
                    $item2->employee_name = $employee_name;

                    $item2->alternative_employee_id = $alternativeEmployee;
                    $item2->alternative_employee_name = $alternative_employee_name;
                } else {
                    $employee_name = SchoolEmployee::findOrFail($item2->school_employee_id)->name;
                    $item2->employee_name = $employee_name;
                }

                $schoolStudentsCount = Student::where('school_id', $request->school_id)->where('class_id', $item2['school_class_id'])->count();
                $attendedSchoolStudents = PeriodStudentAttendance::where('school_id', $request->school_id)->where('period_id', $item->id)->where('semester_id', $semester->id)->where('class_id', $item2['school_class_id'])->where('status', 'attended')->where('date', $date)->count();
                $absentSchoolStudents = PeriodStudentAttendance::where('school_id', $request->school_id)
                    ->where('period_id', $item->id)
                    ->where('class_id', $item2['school_class_id'])
                    ->where('date', $date)
                    ->where('semester_id', $semester->id)
                    ->where(function ($q) {
                        $q->where('status', 'absent')
                            ->orWhere('status', 'absent_with_excuse');
                    })->count();

                $statusMatched = $schoolStudentsCount != 0 && ($schoolStudentsCount == $attendedSchoolStudents + $absentSchoolStudents);
                $class = SchoolClass::findOrFail($item2->school_class_id);

                $item2->subject_name = Subject::find($item2->subject_id)?->name;
                $item2->class_name_id = $class->class_name_id;
                $item2->class_name = $class->className->name;
                $item2->class_number = $class->class_number;
                $item2->date_options = Helper::getDateOptions($date, 'coptic');
                $item2->type = $typeMatched ? 'waiting' : 'none'; // Add 'type'
                $item2->status = $statusMatched ? 'attended' : 'not_attended'; // Add 'status'

                return $item2; // Return the complete array
            })->unique(function ($item2) {
                return $item2['id'];
            });
            $teacherSchedulePeriods->setCollection(collect($periodDetails));
            $periodDetails = $teacherSchedulePeriods;

            $RegisteredClasses = PeriodStudentAttendance::select('class_id')
                ->where('school_id', $request->school_id)
                ->where('date', $date)
                ->where('semester_id', $semester->id)
                ->where('period_id', $item->id)
                ->whereNotNull('status')
                ->groupBy('class_id')
                ->get();
            $AllClasses = SchoolClass::where('school_id', $request->school_id)->count();

            $item->RegisteredClasses = count($RegisteredClasses);
            $item->AllClasses = $AllClasses;
            $item->periodDetails = $periodDetails;

            return $item; // Return the complete array
        });

        return response()->json([
            'datePeriods' => $finalPeriods,
            'schoolStudentsCount' => $schoolStudentsCount,
            'attendedSchoolStudents' => $attendedSchoolStudents,
            'absentSchoolStudents' => $absentSchoolStudents,
            'message' => trans(
                'api.list of date available periods'
            )
        ]);
    }

    public function getDayPeriodStudents(Request $request)
    {
        // Validate incoming request (optional but recommended)
        $request->validate([
            'school_id' => 'required|exists:schools,id',
            'period_id' => 'required|exists:periods,id',
            'class_name_id' => 'required|exists:class_names,id',
            'class_number' => 'required',
            'date_type' => 'nullable|in:coptic,hijri',
        ]);

        if ($request->date) {
            if ($request->date_type == 'hijri') {
                $date = Helper::getCoptic($request->date);
            } else {
                $date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->date)->format('d/m/Y');
            }
        } else {
            $date = date('d/m/Y');
        }

        $semester = Helper::getCurrentSchoolSemester($date);
        if (!$semester) {
            return response()->json(['error' => trans('api.there is no semester registered for this date')], 404);
        }


        $class = SchoolClass::where('school_id', $request->school_id)->where('class_name_id', $request->class_name_id)->where('class_number', $request->class_number)->first();
        if (!$class) {
            return response()->json(['error' => trans('api.there is no class registered for this data')], 404);
        }

        $studentIds = Student::where('school_id', $request->school_id)->where('class_id', $class->id);

        if ($request->student_name) {
            $studentIds->where('name', 'LIKE', "{$request->student_name}%");
        }

        $studentIds = $studentIds->pluck('id')->toArray();

        $checkIfCreated = PeriodStudentAttendance::where('date', $date)->where('semester_id', $semester->id)->where('class_id', $class->id)->where('period_id', $request->period_id)->where('school_id', $request->school_id)->exists();
        if (!$checkIfCreated) {
            foreach ($studentIds as $student_id) {
                $studentAttendance = new PeriodStudentAttendance();
                $studentAttendance->school_id = $request->school_id;
                $studentAttendance->semester_id = $semester->id;
                $studentAttendance->student_id = $student_id;
                $studentAttendance->class_id = Student::findOrFail($student_id)?->class_id;
                $studentAttendance->period_id = $request->period_id;
                $studentAttendance->status = 'attended';
                $studentAttendance->date = $date;
                $studentAttendance->date_type = 'coptic';
                $studentAttendance->save();
            }
        }

        // Paginate results with a default value if not provided
        $perPage = $request->per_page ?? 15;

        $studentAttendance = PeriodStudentAttendance::whereIn('student_id', $studentIds)
            ->where('class_id', $class->id)
            ->where('semester_id', $semester->id)
            ->where('period_id', $request->period_id)
            ->where('school_id', $request->school_id)
            ->where('date', 'LIKE', $date)
            ->paginate($perPage);
        $studentAttendances = StudentAttendanceResource::collection($studentAttendance);

        // All Top Statistics
        $schoolStudentsCount = count($studentIds);
        $attendedSchoolStudents = PeriodStudentAttendance::where('school_id', $request->school_id)
            ->where('semester_id', $semester->id)
            ->where('period_id', $request->period_id)
            ->where('status', 'attended')->where('date', $date)
            ->count();
        $absentSchoolStudents = PeriodStudentAttendance::where('school_id', $request->school_id)
            ->where('date', $date)
            ->where('semester_id', $semester->id)
            ->where('period_id', $request->period_id)
            ->where(function ($q) {
                $q->where('status', 'absent')
                    ->orWhere('status', 'absent_with_excuse');
            })->count();
        $StudentsNotRegisteredYet = $schoolStudentsCount - ($attendedSchoolStudents + $absentSchoolStudents);
        $StudentsNotRegisteredYet = $StudentsNotRegisteredYet > 0 ? $StudentsNotRegisteredYet : 0;
        $RegisteredClasses = PeriodStudentAttendance::select('class_id')
            ->where('school_id', $request->school_id)
            ->where('semester_id', $semester->id)
            ->where('period_id', $request->period_id)
            ->where('date', $date)
            ->whereNotNull('status')
            ->groupBy('class_id')
            ->get();
        $AllClasses = SchoolClass::where('school_id', $request->school_id)->count();

        return response()->json([
            'studentAttendances' => $studentAttendances,
            'schoolStudentsCount' => $schoolStudentsCount,
            'attendedSchoolStudentsCount' => $attendedSchoolStudents,
            'absentSchoolStudentsCount' => $absentSchoolStudents,
            'StudentsNotRegisteredYetCount' => $StudentsNotRegisteredYet,
            'RegisteredClassesCount' => count($RegisteredClasses),
            'AllClassesCount' => $AllClasses,
            'message' => trans(
                'api.list of students attendance'
            )
        ]);
    }

    /** autocomplete the school Student name */
    public function studentAutocompleteName(Request $request)
    {
        $results = Student::where('school_id', $request->id)->where('name', 'LIKE', "{$request->student_name}%")
            ->select('id', 'name')
            ->get();
        return response()->json($results);
    }

    /** change status of attendance to particular employee */
    public function changePeriodAttendanceStatus(Request $request)
    {
        // Validate incoming request
        $rules = [
            'school_id' => 'required|exists:schools,id',
            'period_id' => 'required|exists:periods,id',
            'class_name_id' => 'required|exists:class_names,id',
            'class_number' => 'required',
            'date_type' => 'nullable|in:coptic,hijri',
            'records' => 'required|array|min:1', // Ensures records is an array and has at least 1 entry
            'records.*.student_id' => 'required|exists:school_students,id',
            'records.*.status' => 'required|in:attended,absent,absent_with_excuse',
        ];

        // Add specific validation for 'absence_reason' if 'status' is 'absent_with_excuse'
        // if ($request->input('records')) {
        //     foreach ($request->input('records') as $index => $record) {
        //         if (isset($record['status']) && $record['status'] === 'absent_with_excuse') {
        //             $rules["records.$index.absence_reason"] = 'required|string|max:140';
        //         }
        //     }
        // }

        $request->validate($rules);

        if ($request->date) {
            if ($request->date_type == 'hijri') {
                $date = Helper::getCoptic($request->date);
            } else {
                $date = $request->date;
            }
        } else {
            $date = date('d/m/Y');
        }

        $semester = Helper::getCurrentSchoolSemester($date);
        if (!$semester) {
            return response()->json(['error' => trans('api.there is no semester registered for this date')], 404);
        }

        $class = SchoolClass::where('school_id', $request->school_id)
            ->where('class_name_id', $request->class_name_id)
            ->where('class_number', $request->class_number)
            ->first();

        if (!$class) {
            return response()->json(['error' => trans('api.class not found')], 404);
        }

        foreach ($request->input('records') as $record) {
            $studentAttendance = PeriodStudentAttendance::where('school_id', $request->school_id)
                ->where('semester_id', $semester->id)
                ->where('class_id', $class->id)
                ->where('period_id', $request->period_id)
                ->where('student_id', $record['student_id'])
                ->where('date', 'LIKE', "%$date%")
                ->firstOrFail();

            $studentAttendance->update([
                'status' => $record['status'],
                'absence_reason' => $record['absence_reason'] ?? null,
            ]);
        }

        return response()->json(['message' => trans('api.student attendance updated successfully')]);
    }

    /** تغييب الكل */
    public function makeAllPeriodStudentsAbsent(Request $request)
    {
        // Validate incoming request (optional but recommended)
        $request->validate([
            'school_id' => 'required|exists:schools,id',
            'class_name_id' => 'required|exists:class_names,id',
            'class_number' => 'required',
            'period_id' => 'required|exists:periods,id',
            'date_type' => 'nullable|in:coptic,hijri',
        ]);

        if ($request->date) {
            if ($request->date_type == 'hijri') {
                $date = Helper::getCoptic($request->date);
            } else {
                $date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->date)->format('d/m/Y');
            }
        } else {
            $date = date('d/m/Y');
        }

        $semester = Helper::getCurrentSchoolSemester($date);
        if (!$semester) {
            return response()->json(['error' => trans('api.there is no semester registered for this date')], 404);
        }

        $class = SchoolClass::where('school_id', $request->school_id)
            ->where('class_name_id', $request->class_name_id)
            ->where('class_number', $request->class_number)
            ->first();

        if (!$class) {
            return response()->json(['error' => trans('api.class not found')], 404);
        }

        $studentAttendances = PeriodStudentAttendance::where('school_id', $request->school_id)
            ->where('date', $date)
            ->where('class_id', $class->id)
            ->where('semester_id', $semester->id)
            ->where('period_id', $request->period_id)
            ->get();

        foreach ($studentAttendances as $studentAttendance) {
            $studentAttendance->update([
                'status' => 'absent',
                'absence_reason' => null,
            ]);
        }
        return response()->json(['message' => trans('api.all student Attendance updated to Absent Successfully')]);
    }
}
