<?php

namespace App\Http\Controllers\Api\ManagerDashboard\Settings;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Artisan;
use App\Models\ManagerDashboard\Backup;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Services\DatabaseSwitcher;

class DataArchivingController extends Controller
{
    /* to uses the class methods
        -you should be authenticated
        -you should be authorized and have this permission
    */

    private $databaseSwitcher;

    public function __construct(DatabaseSwitcher $databaseSwitcher)
    {
        $this->databaseSwitcher = $databaseSwitcher;
    }

    public function index(Request $request)
    {
        // return 'Current Database: ' . DB::connection()->getDatabaseName();
        $request->validate([
            'type' => 'required|in:full,partial',
        ]);

        $perPage = $request->input('per_page', 10);

        if($request->type == 'full'){
            $backups = Backup::where('type', 'full')->orderBy('created_at', 'desc')->paginate($perPage);
        } else {
            $backups = Backup::where('type', 'partial')->orderBy('created_at', 'desc')->paginate($perPage);
        }

        return response()->json([
            'data' => $backups,
            'message' => trans('api.list of all archive files'),
        ], 200);
    }

    // Manually create a backup (Full or Partial)
    public function createBackup(Request $request)
    {
        $type = $request->input('type'); // 'full' or 'partial'

        if (!in_array($type, ['full', 'partial'])) {
            return response()->json(['error' => 'Invalid backup type.'], 400);
        }

        // Trigger the backup creation command
        Artisan::call('backup:create', ['type' => $type]);

        return response()->json(['message' => ucfirst($type) . ' backup created successfully.'], 201);
    }

    public function previewBackup($id)
    {
        try {
            $backup = Backup::findOrFail($id);
            $this->databaseSwitcher->switchToArchive($backup);

            return response()->json([
                'message' => 'Preview loaded successfully',
                // 'database_info' => $this->databaseSwitcher->getDatabaseInfo()
            ]);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    public function endPreview()
    {
        try {
            $this->databaseSwitcher->switchToMain();

            return response()->json([
                'message' => 'Preview ended successfully',
                // 'database_info' => $this->databaseSwitcher->getDatabaseInfo()
            ]);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }


    // Download a backup
    public function downloadBackup($id)
    {
        $backup = Backup::findOrFail($id);
        $filePath = storage_path("app/backups/{$backup->name}.sql");

        if (!file_exists($filePath)) {
            return response()->json(['error' => 'File not found.'], 404);
        }

        return response()->download($filePath);
    }

    // Delete a backup
    public function deleteBackup($id)
    {
        $backup = Backup::findOrFail($id);
        $filePath = storage_path("app/backups/{$backup->name}.sql");
        if (file_exists($filePath)) {
            unlink($filePath);
        }

        $backup->delete();

        return response()->json(['message' => 'Backup deleted successfully.']);
    }
}
