<?php

namespace App\Http\Controllers\Api\ManagerDashboard\Exams\StudentCommitteesold;

use App\Http\Controllers\Controller;
use App\Http\Resources\ManagerDashboard\Exams\StudentCommittes\CommitteeClassDataResource;
use App\Http\Resources\ManagerDashboard\Exams\StudentCommittes\CommitteeDataResource;
use App\Http\Resources\ManagerDashboard\Exams\StudentCommittes\StudentCommitteeResource;
use App\Models\AdminDashboard\School\School;
use App\Models\AdminDashboard\School\SchoolClass;
use App\Models\AdminDashboard\School\SchoolSemester;
use App\Models\ManagerDashboard\Exams\StudentCommittees\StudentCommittee;
use App\Models\ManagerDashboard\Exams\StudentCommittees\StudentSeatingNumber;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Alkoumi\LaravelHijriDate\Hijri;
use App\Http\Requests\ManagerDashboard\Exams\StudentCommittes\StoreCommitteeRequest;
use App\Models\AdminDashboard\School\Student;
use App\Models\ManagerDashboard\Exams\StudentCommittees\StudentCommitteeData;
use App\Models\ManagerDashboard\Namazeg\NamazegVersion;
use App\Models\Namazeg;
use App\Traits\NamazegTrait;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use function PHPUnit\Framework\isEmpty;

class StudentCommitteeController extends Controller
{

    use  NamazegTrait;
    /**
     * Display a listing of the resource.
     */

    public function index(School $school)
    {
        // Retrieve school semesters with related student committees
        $schoolSemesters = SchoolSemester::with([
            'exam_student_committees' => function ($query) use ($school) {
                $query->where('school_id', $school->id);
            }
        ])->where('status','1')->get();

        // Prepare the response
        $response = $schoolSemesters->map(function ($semester) {
            // Group committees by type for this semester
            $groupedCommittees = $semester->exam_student_committees->groupBy('committee_student_type');

            return [
                'semester_id' => $semester->id,
                'semester_name' => $semester->name,
                'committees' => $groupedCommittees->map(function ($committees) {
                    return [
                        'committees' => $committees->map(function ($studentCommittee) {
                            return [
                                'committe_id' => $studentCommittee->id,
                                'name' => $studentCommittee->name,
                                'serial_number' => $studentCommittee->namozegVersion?->serial_number,
                                'status' => $studentCommittee->status,
                                'schoolclass_id' => $studentCommittee->schoolclass_id,

                            ];
                        }),
                    ];
                }),
            ];
        });

        return response()->json(['schoolSemesters' => $response]);
    }

    public function create(School $school, SchoolSemester $semester)
    {
        // Get the number of remaining committees
        $committeesNumber = StudentCommittee::where('school_id', $school->id)
            ->where('semester_id', $semester->id)
            ->count();

        $distinctCommitteesCount = StudentCommitteeData::whereHas('studentCommittee', function ($query) use ($school, $semester) {
            $query->where('school_id', $school->id)
                ->where('semester_id', $semester->id);
        })
            ->distinct('committee_id')
            ->count('committee_id');

        $remaining_committees = $committeesNumber - $distinctCommitteesCount;

        // Load school classes with class names
        $schoolClasses = SchoolClass::where('school_id', $school->id)
            ->with('className')
            ->get();

        // Fetch committee data filtered by school_id and semester_id
        $committeeData = StudentCommitteeData::with(['studentCommittee'])
            ->whereHas('studentCommittee', function ($query) use ($school, $semester) {
                $query->where('school_id', $school->id)
                    ->where('semester_id', $semester->id);
            })
            ->get();

        // Initialize an array to count students in committee data by class
        $classStudentCounts = [];

        // Process committee data to count students per class
        foreach ($committeeData as $data) {
            $studentIds = json_decode($data->student_ids); // Get student IDs from JSON
            $classId = $data->schoolclass_id; // Get the class ID

            // Count students in this committee data
            if (!isset($classStudentCounts[$classId])) {
                $classStudentCounts[$classId] = 0;
            }
            $classStudentCounts[$classId] += count($studentIds);
        }

        // Initialize final counts
        $finalClassCounts = [];
        $totalRemainingStudents = 0; // Initialize total remaining students

        // Calculate remaining students for each class and retrieve student details
        foreach ($schoolClasses as $class) {
            $className = $class->className->name;
            $totalStudentsInClass = Student::where('class_id', $class->id)->count();

            // Get students from the current class
            $students = Student::with('schoolClass')->where('class_id', $class->id)
                ->get(['id', 'name', 'class_id']);
            // Get count from committee data or set to 0 if not present
            $studentsInCommittee = $classStudentCounts[$class->id] ?? 0;

            // Calculate remaining students for this class
            $remainingStudents = $totalStudentsInClass - $studentsInCommittee;

            // Update total remaining students
            $totalRemainingStudents += max(0, $remainingStudents);

            // Store in final counts, ensuring we group by class name
            if (!isset($finalClassCounts[$className])) {
                $finalClassCounts[$className] = [
                    'class_name' => $className,
                    'remaining_students' => max(0, $remainingStudents),
                    'students' => $students->map(function ($student) {
                        return [
                            'id' => $student->id,
                            'name' => $student->name,
                            'class_number' => $student->schoolClass->class_number,
                        ];
                    })->toArray(),
                ];
            } else {
                // Aggregate the counts if the class name already exists
                $finalClassCounts[$className]['remaining_students'] += max(0, $remainingStudents);
                $finalClassCounts[$className]['students'] = array_merge(
                    $finalClassCounts[$className]['students'],
                    $students->map(function ($student) {
                        return [
                            'id' => $student->id,
                            'name' => $student->name,
                            'class_number' => $student->schoolClass->class_number,
                        ];
                    })->toArray()
                );
            }
        }

        return response()->json([
            'remaining_committees' => $remaining_committees,
            'total_students' => max(0, $totalRemainingStudents), // This reflects remaining students in the school
            'class_student_counts' => array_values($finalClassCounts), // Return the final counts
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreCommitteeRequest $request, School $school)
    {
        $data = $request->validated();

        try {
            $namozeg = Namazeg::where('form_number', 61)->firstOrFail();
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'Namozeg Not Found for this committee form number in namazeg table .'], 404);
        }


        // لجان اختبارات من نفس الصف
        if (!isset($data['commitees_number']) && !isset($data['student_type'])) {
            $schoolClasses = SchoolClass::where('school_id', $school->id)->get();
            $groupedClasses = $schoolClasses->groupBy('class_name_id');

            foreach ($groupedClasses as  $groupedClass) {
                foreach ($groupedClass as $class) {

                    $className = $class->className->name;
                    $classNumber = $class->class_number;

                    // // Get the Arabic class number from the model
                    // $arabicClassNumber = SchoolClass::getArabicClassNumber($classNumber);

                    // // Concatenate for the full name
                    // $fullName = trim("{$className} الفصل {$arabicClassNumber}");

                   // Get the last committee for the school
                    $lastCommittee = StudentCommittee::where('school_id', $school->id)->orderBy('id', 'desc')->first();

                    // Default to 1 if no committee exists yet
                    if ($lastCommittee) {
                        // Use regular expression to extract the number from the name
                        preg_match('/\d+$/', $lastCommittee->name, $matches);
                        $committeeNumber = $matches ? (intval($matches[0]) + 1) : 1;  // Increment the number or default to 1 if no match
                    } else {
                        $committeeNumber = 1;  // No committees found, so this will be the first committee
                    }

                    // Create the committee name dynamically
                    $committeeName = "اللجنة " . $committeeNumber;

                    $students = $class->Students()->where('school_id', $school->id)->get();

                    // If you want to get just the IDs
                    $studentIds = $students->pluck('id'); // Get an array of student IDs

                    $current_year = substr(Hijri::Date('Y'), -2);


                    // If no version exists, create a new one using the trait method
                    $namozeg_version = $this->storeNamazegVersion(
                        $namozeg->id,
                        $namozeg->internal_extenral,
                        $school->id
                    );

                     // Check if a committee already exists for the given school and semester
                            $existingCommittee = StudentCommittee::where('school_id', $school->id)
                            ->where('semester_id', $data['semester_id'])
                            ->first();

                        if ($existingCommittee) {
                            // If a committee already exists for the given school and semester, return an error
                            return response()->json([
                                'error' => 'A committee already exists for this school in the selected semester.'
                            ], 400); // HTTP 400 Bad Request
                        }

                    $committee =  StudentCommittee::create(
                        [
                            'namazeg_version_id' => $namozeg_version->id,
                            'school_id' => $school->id,
                            'semester_id' =>  $data['semester_id'],
                            'schoolclass_id' =>  $class->id,
                            'name' => $committeeName,
                            'committee_student_type' => $class->entsab === 'no'  ? 'antzam' : 'entsab',
                            'status' => 'complated'
                        ]
                    );

                    foreach ($studentIds as $studentId) {

                        // Create a base for seat number
                        // $baseSeatNumber = "{$current_year}{$class->id}{$classNumber}{$studentId}";
                        $baseSeatNumber = "{$current_year}{$class->className->id}{$classNumber}{$studentId}";

                        // Check if the student already has a seat number in the committee
                        $existingSeating = StudentSeatingNumber::where('semester_id', $committee->semester->id)
                            ->where('student_id', $studentId)
                            ->where('school_id', $school->id)
                            ->first();

                        if ($existingSeating) {
                            // Return an error message if the student already exists in the committee with a seat number
                            return response()->json([
                                'error' => "Student with ID {$studentId} already exists in the committee and has a seat number."
                            ], 400); // HTTP 400 Bad Request
                        }

                        StudentSeatingNumber::create([
                            'school_id' =>  $school->id,
                            'semester_id' =>  $committee->semester->id,
                            'student_id' => $studentId,
                            'committee_id' => $committee->id,
                            'seat_number' =>  $baseSeatNumber
                        ]);
                    }
                }
            }

            // لجان اختبارات يدوية
        } else {

            for ($i = 0; $i < $data['commitees_number']; $i++) {

                // If no version exists, create a new one using the trait method
                $namozeg_version = $this->storeNamazegVersion(
                    $namozeg->id,
                    $namozeg->internal_extenral,
                    $school->id
                );

                StudentCommittee::create(
                    [
                        'namazeg_version_id' => $namozeg_version->id,
                        'school_id' => $school->id,
                        'semester_id' =>  $data['semester_id'],
                        'name' => 'اللجنة ' . ($i + 1),
                        'committee_student_type' =>  $data['committee_student_type'],
                        'status' => 'incomplated'

                    ]
                );
            }
        }

        return response()->json(['message' => trans('api.record stored successflly')]);
    }

    /**
     * Display the specified resource.
     */
    public function show(School $school, StudentCommittee $committee)
    {
        $committeeData  = new CommitteeDataResource($committee->load(['SchoolClass', 'CommitteeData', 'CommitteeStudents.student', 'CommitteeStudents.committeeData.SchoolClass']));

        return response()->json(['committeeData' => $committeeData]);
    }


    /**
     * Remove the specified resource from storage.
     */
    public function destroy(School $school, StudentCommittee $committee)
    {
        StudentCommittee::where('school_id', $school->id)->where('id', $committee->id)->forceDelete();

        return response()->json(['message' => trans('api.record deleted successflly')]);
    }

    public function deleteAllCommittees(School $school, SchoolSemester $semester)
    {
        StudentCommittee::where('school_id', $school->id)->where('semester_id', $semester->id)->forceDelete();

        return response()->json(['message' => 'تم حذف جميع اللجان لهذا الفصل الدراسي بنجاح']);
    }

    public function getExamApplicantsCount(School $school)
    {

        $entsabStudents = Student::where('school_id', $school->id)
            ->whereHas('schoolClass', function ($query) {
                $query->where('entsab', 'no');
            })
            ->count();

        $regularStudents = Student::where('school_id', $school->id)
            ->whereHas('schoolClass', function ($query) {
                $query->where('entsab', 'yes');
            })
            ->count();
      
        return response()->json(['entsabStudents' => $entsabStudents,'regularStudents'=>$regularStudents]);

    }
}
