<?php

namespace App\Http\Controllers\Api\ManagerDashboard\Assignments;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use App\Http\Requests\ManagerDashboard\Assignments\AssignmentTeamRequest;
use App\Http\Resources\ManagerDashboard\Assignments\AssignmentTeamDetails;
use App\Models\AdminDashboard\School\School;
use App\Models\ManagerDashboard\Assignments\AssignmentTeam;
use App\Models\ManagerDashboard\Assignments\TaskCommitteeTeam;
use App\Models\ManagerDashboard\Namazeg\NamazegVersion;
use App\Models\ManagerDashboard\TeamAndCommitteeMeetings\Meeting;
use App\Models\ManagerDashboard\TeamAndCommitteeMeetings\Team;
use App\Models\ManagerDashboard\TeamAndCommitteeMeetings\TeamMember;
use App\Models\Namazeg;
use App\Traits\NamazegTrait;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AssignmentTeamController extends Controller
{
    use NamazegTrait;

    /**
     * Store a newly created resource in storage.
     */
    public function storeOrUpdate(School $school, Team $team, AssignmentTeamRequest $request)
    {
        // Start a database transaction
        DB::beginTransaction();

        try {
            $data = $request->validated();


            $teamId = $team->id;

            $namozeg = Namazeg::firstWhere('form_number', $team->assignment_form_number);

            // Check if a NamazegVersion already exists for the school and namozeg_id
            $existing_version = NamazegVersion::where('namazeg_id', $namozeg->id)
                ->where('school_id', $school->id)
                ->first();

            if (!$existing_version) {
                // If no version exists, create a new one using the trait method
                $namozeg_version = $this->storeNamazegVersion(
                    $namozeg->id,
                    $namozeg->internal_extenral,
                    $school->id
                );
            } else {
                // Use the existing version
                $namozeg_version = $existing_version;
            }

            $date = Carbon::createFromFormat('d/m/Y', $data['assignment']['date'])->format('d/m/Y');
            $date_type = $data['assignment']['date_type'];

            if ($date_type === 'hijri') {
                $date = Helper::getCoptic($date);
                $date_type = 'coptic';
            }

            // Store Assignment 
            $assignmentTeam = AssignmentTeam::updateOrCreate(
                [
                    // 'namazeg_version_id' => $namozeg_version->id, // Version of Namazeg for this school
                    'school_id' => $school->id, // Ensure it’s for this school
                    'team_id' => $teamId // Ensure it’s for the same team
                ],
                [
                    'namazeg_version_id' => $namozeg_version->id, // Version of Namazeg for this school
                    'date' => $date,
                    'date_type' => $date_type
                ]
            );

            // Delete existing Team Members before re-inserting
            TeamMember::where('team_id', $teamId)
                ->where('assignment_team_id', $assignmentTeam->id)
                ->forceDelete();

            // Store Team Members
            foreach ($data['team_members'] as $team_member) {
                TeamMember::updateOrCreate(
                    [
                        'team_id' => $teamId,
                        'assignment_team_id' => $assignmentTeam->id,
                        'employee_id' => $team_member['employee_id'],
                        'assignment_work' => $team_member['assignment_work']
                    ]
                );
            }

            // Delete existing Mettings before re-inserting
            Meeting::where('school_id', $school->id)
                ->where('model_id', $assignmentTeam->team_id)
                ->where('meeting_type', 'periodic')
                ->forceDelete();

            // Store Tasks
            $team->tasks()->where('school_id',$school->id)->delete();

            if (!empty($data['tasks'])) {

                foreach ($data['tasks'] as $task) {
                    $task = new TaskCommitteeTeam([
                        'school_id' => $school->id,
                        'task_name' => $task,

                    ]);
                    $team->tasks()->save($task);
                }
            }

            // Store Team Meetings

            $meetings = [];

            foreach ($data['team_semesters_meetings'] as $team_semester_meetings) {

                $semesterId = $team_semester_meetings['semester_id'];

                foreach ($team_semester_meetings['meetings'] as $team_semester_meeting) {
                    $meeting_date = Carbon::createFromFormat('d/m/Y', $team_semester_meeting['date'])->format('d/m/Y');
                    $meeting_date_type = $team_semester_meeting['date_type'];

                    if ($meeting_date_type === 'hijri') {
                        $meeting_date = Helper::getCoptic($meeting_date);
                        $meeting_date_type = 'coptic';
                    }

                    $meetings[] = [
                        'school_id' => $school->id,
                        'semester_id' => $semesterId,
                        'model_type' => 'teams',
                        'model_id' => $teamId,
                        'name' => $team_semester_meeting['name'],
                        'date' => $meeting_date,
                        'date_type' => $meeting_date_type,
                        'place' => $team_semester_meeting['place'] ?? null,
                        'meeting_type' => 'periodic',
                        'meeting_status' => 'not_completed'
                    ];
                }
            }

            // Batch insert all meetings
            Meeting::insert($meetings);

            // Commit the transaction
            DB::commit();

            return response()->json(['message' => trans('api.record stored successfully')], 200);
        } catch (\Exception $e) {
            // Rollback the transaction in case of an error
            DB::rollBack();
            dd($e->getMessage());
            // Return an error response
            return response()->json(['message' => 'حدث خطأ، الرجاء المحاولة في وقت لاحق'], 500);
        }
    }

    public function details(School $school, AssignmentTeam $assignment)
    {
       
     
        $assignment = AssignmentTeam::with([
            'team.tasks' => function ($query) use ($school) {
                $query->where('school_id', $school->id); 
            },
            'team.meetings' => function ($query) use ($school) {
                $query->where('school_id', $school->id); 
            },
            'teamMembers' => function ($query) use ($assignment) {
                $query->where('assignment_team_id', $assignment->id); 
            },
        ])
        ->find($assignment->id); // Ensure you're retrieving the correct assignment by ID
      
        return new AssignmentTeamDetails($assignment);

    }

    /**
     * Soft Delete.
     */
    public function archive(School $school, Team $team)
    {

        DB::transaction(function () use ($school, $team) {
            // Get the specific assignment for the team and the school
            $assignmentTeam = AssignmentTeam::where('team_id', $team->id)
                ->where('school_id', $school->id)
                ->firstOrFail();

            // Delete AssignmentTeam specific to this school
            if ($assignmentTeam) {
                // Delete the assignment team members related to this specific assignment
                TeamMember::where('assignment_team_id', $assignmentTeam->id)
                    ->where('team_id', $team->id)
                    ->delete();

                // Delete the assignment team itself
                $assignmentTeam->delete();
            }

            // Delete related meetings, also filtered by the team and school
            Meeting::where('model_type', 'teams')
                ->where('model_id', $team->id)
                ->where('school_id', $school->id)
                ->delete();
        });

        return response()->json(['message' => trans('api.record archived successflly')]);
    }



    /**
     * restore the specified resource which soft Deleted.
     */
    public function restore(School $school, Team $team)
    {
        // Fetch soft-deleted AssignmentTeam, if it exists
        $assignmentTeam = AssignmentTeam::withTrashed()
            ->where('team_id', $team->id)
            ->where('school_id', $school->id)
            ->first();

        if ($assignmentTeam) {
            // Restore AssignmentTeam
            $assignmentTeam->restore();

            // Restore TeamMembers related to this AssignmentTeam
            TeamMember::withTrashed()
                ->where('assignment_team_id', $assignmentTeam->id)
                ->where('team_id', $team->id)
                ->restore();

            // Restore related meetings
            Meeting::withTrashed()
                ->where('model_type', 'teams')
                ->where('model_id', $team->id)
                ->where('school_id', $school->id)
                ->restore();

            return response()->json(['message' => trans('api.record restored successfully')]);
        }

        return response()->json(['message' => trans('api.record restored successflly')]);
    }


    /**
     * Remove the specified resource from storage.
     */
    public function destroy(School $school, Team $team)
    {
        DB::transaction(function () use ($school, $team) {
            // Get the specific assignment for the team and the school
            $assignmentTeam = AssignmentTeam::where('team_id', $team->id)
                ->where('school_id', $school->id)
                ->first();

            // Delete AssignmentTeam specific to this school
            if ($assignmentTeam) {
                // Delete the assignment team members related to this specific assignment
                TeamMember::where('assignment_team_id', $assignmentTeam->id)
                    ->where('team_id', $team->id)
                    ->forceDelete();

                // Delete the assignment team itself
                $assignmentTeam->forceDelete();
            }
            $team->tasks()->where('school_id',$school->id)->delete();
            // Delete related meetings, also filtered by the team and school
            Meeting::where('model_type', 'teams')
                ->where('model_id', $team->id)
                ->where('school_id', $school->id)
                ->forceDelete();
        });

        return response()->json(['message' => trans('api.record deleted successflly')]);
    }
}
