<?php

namespace App\Http\Controllers\Api\ManagerDashboard\Assignments;

use App\Helpers\Helper;
use App\Helpers\HelperNotification;
use App\Http\Controllers\Controller;
use App\Http\Requests\ManagerDashboard\Assignments\StoreAssignmentRequest;
use Illuminate\Http\Request;
use App\Models\ManagerDashboard\Assignments\AssignmentCategory;
use App\Http\Resources\ManagerDashboard\Assignments\AssignmentCategoryResource;
use App\Http\Resources\ManagerDashboard\Assignments\CommitteeAssignmentResource;
use App\Http\Resources\ManagerDashboard\Assignments\GeneralAssignmentResource;
use App\Http\Resources\ManagerDashboard\Assignments\TeamAssignmentResource;
use App\Models\AdminDashboard\School\CurrentWork;
use App\Models\ManagerDashboard\Assignments\Assignment;
use App\Models\ManagerDashboard\Assignments\AssignmentEmployee;
use App\Models\ManagerDashboard\Assignments\GeneralAssignment;
use App\Models\ManagerDashboard\TeamAndCommitteeMeetings\Committee;
use App\Models\ManagerDashboard\TeamAndCommitteeMeetings\Team;
use App\Models\AdminDashboard\School\School;
use App\Models\AdminDashboard\School\SchoolEmployee;
use App\Models\ManagerDashboard\Namazeg\NamazegVersion;
use App\Models\Namazeg;
use App\Traits\LogsActivity;
use App\Traits\NamazegTrait;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class AssignmentController extends Controller
{
    use NamazegTrait;

    use LogsActivity;

    /**
     * Store a newly created resource in storage.
     */
    public function store(School $school, StoreAssignmentRequest $request)
    {
        // Start the transaction
        DB::beginTransaction();

        try {
            $data = $request->validated();

            $currentWorkId  =  $data['current_work_id'];

            $CurrentWork =  CurrentWork::where('id', $currentWorkId)->first();

            $employeeIds = $data['employees'];

            unset($data['employees']);

            $namozeg = Namazeg::firstWhere('form_number', $CurrentWork->assignment_form_number);

            // Check if a NamazegVersion already exists for the school and namozeg_id
            $existing_version = NamazegVersion::where('namazeg_id', $namozeg->id)
                ->where('school_id', $school->id)
                ->first();

            if (!$existing_version) {
                // If no version exists, create a new one using the trait method
                $namozeg_version = $this->storeNamazegVersion(
                    $namozeg->id,
                    $namozeg->internal_extenral,
                    $school->id
                );
            } else {
                // Use the existing version
                $namozeg_version = $existing_version;
            }

            $date = Carbon::createFromFormat('d/m/Y', $data['date'])->format('d/m/Y');

            $date_type = $data['date_type'];

            if ($date_type === 'hijri') {
                $date = Helper::getCoptic($date);
                $date_type = 'coptic';
            }

            $staffAssignment  = Assignment::updateOrCreate(
                [
                    // 'namazeg_version_id' => $namozeg_version->id,
                    'school_id' => $school->id,
                ],
                [
                    'namazeg_version_id' => $namozeg_version->id,
                    'date' => $date,
                    'date_type' => $date_type,
                    'current_work_id' => $data['current_work_id'],
                ]
            );

            // Delete existing AssignmentEmployee before re-inserting
            // AssignmentEmployee::where('assignment_id', $staffAssignment->id)->forceDelete();

            $duplicateEmployees = [];

            // Store the employees
            foreach ($employeeIds as $employeeId) {

                $existingAssignment = AssignmentEmployee::where('employee_id', $employeeId)->first();

                if ($existingAssignment) {
                    $duplicateEmployees[] = $employeeId;
                    continue; // Skip this employee and move to the next one
                }

                // Check if the employee has a specialization
                $employee = SchoolEmployee::with('currentWork')->find($employeeId);

                if (!$employee || !$employee->specialization_id) {
                    // Return an error response if the employee has no specialization
                    return response()->json([
                        'message' => 'هذا المعلم ' . $employee->name . ' ليس له تخصص، برجاء إضافة التخصص أولاً.'
                    ], 422); // 422 Unprocessable Entity
                }


                AssignmentEmployee::create([
                    'assignment_id' => $staffAssignment->id,
                    'employee_id' => $employeeId
                ]);

                HelperNotification::create_teacher_assignment_notifications($staffAssignment, $employee);
            }

            // If there are duplicate employees, return an error
            if (!empty($duplicateEmployees)) {
                // Fetch the names of the duplicate employees
                $employees = SchoolEmployee::select('name')
                    ->whereIn('id', $duplicateEmployees)
                    ->get()
                    ->pluck('name') // Extract only the 'name' column
                    ->toArray();

                // Rollback the transaction
                DB::rollBack();

                // Return the error message with employee names
                return response()->json([
                    'error' => 'بعض المعلمين مكلفين بالفعل: ' . implode('، ', $employees)
                ], 422);
            }

            // Commit the transaction
            DB::commit();

            // ------------------------------- School ID -----------------------
            $this->ManagerlogActivity('Assignment', 'store', $school->id, $namozeg->form_number);
            return response()->json(['message' => trans('api.record stored successflly')], 200);
        } catch (\Exception $e) {
            // Rollback the transaction in case of an exception
            DB::rollBack();



            // Return a generic error message
            return response()->json(['error' => trans('api.something_went_wrong')], 500);
        }
    }
    /**
     * Soft Delete.
     */
    public function archive(School $school, Assignment $assignment, $employeeId)
    {

        AssignmentEmployee::where('assignment_id', $assignment->id)->where('employee_id', $employeeId)->delete();
        $form_number = $assignment->namozegVersion->namozeg->form_number;
        $this->ManagerlogActivity('Assignment', 'archived', $school->id, $form_number);

        return response()->json(['message' => trans('api.record archived successflly')]);
    }

    /**
     * restore the specified resource which soft Deleted.
     */
    public function restore(School $school, Assignment $assignment, $employeeId)
    {

        AssignmentEmployee::where('assignment_id', $assignment->id)->where('employee_id', $employeeId)->restore();

        $form_number = $assignment->namozegVersion->namozeg->form_number;
        $this->ManagerlogActivity('Assignment', 'restored', $school->id, $form_number);

        return response()->json(['message' => trans('api.record restored successflly')]);
    }


    /**
     * Remove the specified resource from storage.
     */
    public function destroy(School $school, Assignment $assignment, $employeeId)
    {

        AssignmentEmployee::where('assignment_id', $assignment->id)->where('employee_id', $employeeId)->forceDelete();

        // Check if there are any employees left for the assignment

        $assignmentEmployees = $assignment->assignmentEmployees->count();
        // If no employees are left, delete the assignment
        if ($assignmentEmployees === 0) {
            $assignment->forceDelete();
        }
        $form_number = $assignment->namozegVersion?->namozeg?->form_number;
        $this->ManagerlogActivity('Assignment', 'delete', $school->id, $form_number);

        return response()->json(['message' => trans('api.record deleted successflly')]);
    }
}
