<?php

namespace App\Http\Controllers\Api\ManagerDashboard\Assignments;

use App\Helpers\Helper;
use App\Helpers\HelperNotification;
use App\Http\Controllers\Controller;
use App\Http\Requests\ManagerDashboard\Assignments\StoreAssignmentCommitteeRequest;
use App\Http\Resources\ManagerDashboard\Assignments\AssignmentcommitteeDetails;
use App\Http\Resources\ManagerDashboard\Assignments\CommitteeAssignmentResource;
use App\Models\AdminDashboard\School\School;
use App\Models\ManagerDashboard\Assignments\AssignmentCommittee;
use App\Models\ManagerDashboard\Assignments\TaskCommitteeTeam;
use App\Models\ManagerDashboard\Namazeg\NamazegVersion;
use App\Models\ManagerDashboard\TeamAndCommitteeMeetings\Committee;
use App\Models\ManagerDashboard\TeamAndCommitteeMeetings\CommitteeMember;
use App\Models\ManagerDashboard\TeamAndCommitteeMeetings\Meeting;
use App\Models\Namazeg;
use App\Traits\LogsActivity;
use App\Traits\NamazegTrait;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AssignmentCommitteeController extends Controller
{
    use NamazegTrait, LogsActivity;
    /**
     * Store a newly created resource in storage.
     */
    public function storeOrUpdate(School $school, Committee $committee, StoreAssignmentCommitteeRequest $request)
    {
        // Start a database transaction
        DB::beginTransaction();

        try {
            $data = $request->validated();
            $committeeId = $committee->id;

            $namozeg = Namazeg::firstWhere('form_number', $committee->assignment_form_number);

            // Check if a NamazegVersion already exists for the school and namozeg_id
            $existing_version = NamazegVersion::where('namazeg_id', $namozeg->id)
                ->where('school_id', $school->id)
                ->first();

            if (!$existing_version) {
                // If no version exists, create a new one using the trait method
                $namozeg_version = $this->storeNamazegVersion(
                    $namozeg->id,
                    $namozeg->internal_extenral,
                    $school->id
                );
            } else {
                // Use the existing version
                $namozeg_version = $existing_version;
            }

            $date = Carbon::createFromFormat('d/m/Y', $data['assignment']['date'])->format('d/m/Y');
            $date_type = $data['assignment']['date_type'];

            if ($date_type === 'hijri') {
                $date = Helper::getCoptic($date);
                $date_type = 'coptic';
            }

            // Store Assignment 
            $assignmentCommittee = AssignmentCommittee::updateOrCreate(
                [
                    // 'namazeg_version_id' => $namozeg_version->id, // Version of Namazeg for this school
                    'school_id' => $school->id, // Ensure it’s for this school
                    'committee_id' => $committeeId // Ensure it’s for the same committee
                ],
                [
                    'namazeg_version_id' => $namozeg_version->id, // Version of Namazeg for this school
                    'date' => $date,
                    'date_type' => $date_type
                ]
            );

            // Delete existing Committee Members before re-inserting
            CommitteeMember::where('committee_id', $committeeId)
                ->where('assignment_committee_id', $assignmentCommittee->id)
                ->forceDelete();


            // Insert new Committee Members
            foreach ($data['committee_members'] as $committee_member) {
                CommitteeMember::create([
                    'committee_id' => $committeeId,
                    'assignment_committee_id' => $assignmentCommittee->id,
                    'employee_id' => $committee_member['employee_id'],
                    'assignment_work' => $committee_member['assignment_work'],
                    'mission' => $committee_member['mission'] ?? null,
                ]);
            }

             // Store Tasks
             $committee->tasks()->where('school_id',$school->id)->delete();

             if (!empty($data['tasks'])) {
 
                 foreach ($data['tasks'] as $task) {
                     $task = new TaskCommitteeTeam([
                         'school_id' => $school->id,
                         'task_name' => $task,
 
                     ]);
                     $committee->tasks()->save($task);
                 }
             }
            // Delete existing Mettings before re-inserting
            Meeting::where('school_id', $school->id)
                ->where('model_id', $assignmentCommittee->committee_id)
                ->where('meeting_type', 'periodic')
                ->forceDelete();



            // Store Committee Meetings

            $meetings = [];

            foreach ($data['committee_semesters_meetings'] as $committee_semester_meetings) {

                $semesterId = $committee_semester_meetings['semester_id'];
                foreach ($committee_semester_meetings['meetings'] as $committee_semester_meeting) {
                    $meeting_date = Carbon::createFromFormat('d/m/Y', $committee_semester_meeting['date'])->format('d/m/Y');
                    $meeting_date_type = $committee_semester_meeting['date_type'];

                    if ($meeting_date_type === 'hijri') {
                        $meeting_date = Helper::getCoptic($meeting_date);
                        $meeting_date_type = 'coptic';
                    }

                    $meetings[] = [
                        'school_id' => $school->id,
                        'semester_id' => $semesterId,
                        'model_type' => 'committees',
                        'model_id' => $committeeId,
                        'name' => $committee_semester_meeting['name'], // Meeting name should be unique,
                        'date' => $meeting_date, // Update details if meeting exists
                        'date_type' => $meeting_date_type,
                        'place' => $committee_semester_meeting['place'] ?? null,
                        'meeting_type' => 'periodic',
                        'meeting_status' => 'not_completed'
                    ];
                }
            }

            // Batch insert all meetings
            Meeting::insert($meetings);

            // Log activity
            $this->ManagerlogActivity('AssignmentCommittee', 'store', $school->id, $namozeg->form_number);

            // Commit the transaction
            DB::commit();

            return response()->json(['message' => trans('api.record stored successfully')], 200);
        } catch (\Exception $e) {

            // Rollback the transaction in case of an error
            DB::rollBack();


            // Return an error response
            return response()->json(['message' => 'حدث خطأ، الرجاء المحاولة في وقت لاحق'], 500);
        }
    }


    public function details(School $school, AssignmentCommittee $assignment)
    {
       
      
        $assignment = AssignmentCommittee::with([
            'committee.tasks' => function ($query) use ($school) {
                $query->where('school_id', $school->id); 
            },
            'committee.meetings' => function ($query) use ($school) {
                $query->where('school_id', $school->id); 
            },
            'CommitteeMember' => function ($query) use ($assignment) {
                $query->where('assignment_committee_id', $assignment->id); 
            },
        ])
        ->find($assignment->id); // Ensure you're retrieving the correct assignment by ID
        return new AssignmentcommitteeDetails($assignment);

    }
    

    /**
     * Soft Delete.
     */
    public function archive(School $school, Committee $committee)
    {
        DB::transaction(function () use ($school, $committee) {
            // Get the specific assignment for the committee and the school
            $assignmentCommittee = AssignmentCommittee::where('committee_id', $committee->id)
                ->where('school_id', $school->id)
                ->firstOrFail();
            $form_number = $assignmentCommittee->namozegVersion->namozeg->form_number;

            // Delete AssignmentCommittee specific to this school
            if ($assignmentCommittee) {
                // Delete the assignment committee members related to this specific assignment
                CommitteeMember::where('assignment_committee_id', $assignmentCommittee->id)
                    ->where('committee_id', $committee->id)
                    ->delete();

                // Delete the assignment committee itself
                $assignmentCommittee->delete();
            }

            // Delete related meetings, also filtered by the committee and school
            Meeting::where('model_type', 'committees')
                ->where('model_id', $committee->id)
                ->where('school_id', $school->id)
                ->delete();
            $this->ManagerlogActivity('AssignmentCommittee', 'archived', $school->id, $form_number);
        });


        return response()->json(['message' => trans('api.record archived successflly')]);
    }



    /**
     * restore the specified resource which soft Deleted.
     */
    public function restore(School $school, Committee $committee)
    {
        // Fetch soft-deleted AssignmentCommittee, if it exists
        $assignmentCommittee = AssignmentCommittee::withTrashed()
            ->where('committee_id', $committee->id)
            ->where('school_id', $school->id)
            ->firstOrFail();

        if ($assignmentCommittee) {
            // Restore AssignmentCommittee
            $assignmentCommittee->restore();

            // Restore CommitteeMembers related to this AssignmentCommittee
            CommitteeMember::withTrashed()
                ->where('assignment_committee_id', $assignmentCommittee->id)
                ->where('committee_id', $committee->id)
                ->restore();

            // Restore related meetings
            Meeting::withTrashed()
                ->where('model_type', 'committees')
                ->where('model_id', $committee->id)
                ->where('school_id', $school->id)
                ->restore();

            $form_number = $assignmentCommittee->namozegVersion->namozeg->form_number;
            $this->ManagerlogActivity('AssignmentCommittee', 'restored', $school->id, $form_number);

            return response()->json(['message' => trans('api.record restored successfully')]);
        }

        return response()->json(['message' => trans('api.no records to restore')], 404);
    }


    /**
     * Remove the specified resource from storage.
     */


    public function destroy(School $school, Committee $committee)
    {
        DB::transaction(function () use ($school, $committee) {
            // Get the specific assignment for the committee and the school
            $assignmentCommittee = AssignmentCommittee::where('committee_id', $committee->id)
                ->where('school_id', $school->id)
                ->firstOrFail();
            // Delete AssignmentCommittee specific to this school
            if ($assignmentCommittee) {
                // Delete the assignment committee members related to this specific assignment
                CommitteeMember::where('assignment_committee_id', $assignmentCommittee->id)
                    ->where('committee_id', $committee->id)
                    ->forceDelete();

                // Delete the assignment committee itself
                $assignmentCommittee->forceDelete();
            }

            $committee->tasks()->where('school_id',$school->id)->delete();
            
            // Delete related meetings, also filtered by the committee and school
            Meeting::where('model_type', 'committees')
                ->where('model_id', $committee->id)
                ->where('school_id', $school->id)
                ->forceDelete();
            $form_number = $assignmentCommittee->namozegVersion->namozeg->form_number;
            $this->ManagerlogActivity('AssignmentCommittee', 'delete', $school->id, $form_number);
        });

        return response()->json(['message' => trans('api.record deleted successfully')]);
    }
}
