<?php

namespace App\Http\Controllers\Api\AdminDashboard\Products;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use App\Http\Requests\AdminDashboard\Products\ProductRequest;
use App\Http\Resources\AdminDashboard\Products\ProductRescource;
use App\Models\AdminDashboard\Products\Product;
use App\Traits\LogsActivity;
use Illuminate\Http\Request;

class ProductController extends Controller
{
    use LogsActivity;
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {

        // Start with the base query
        $data = Product::query();
        // Retrieve all query parameters
        $queryParams = $request->query();
        // Filter data is directly from query parameters, excluding `per_page`
        $filterData = array_filter($queryParams, function ($key) {
            return $key !== 'per_page';
        }, ARRAY_FILTER_USE_KEY);

        // Apply filters if any
        if (!empty($filterData)) {
            $data = Helper::filterData('App\Models\AdminDashboard\Products\Product', $filterData);
        }
        // Paginate results with a default value if not provided
        $perPage = $queryParams['per_page'] ?? 15;

        $filterData = ProductRescource::collection($data->paginate($perPage));


        return response()->json([
            'products' => $filterData,
            'message' => trans('api.list of all available records'),
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(ProductRequest $request)
    {
        $data = $request->validated();
        $product = Product::create($data);
        $this->AdminlogActivity('Product', 'store', '/products/' . $product->id . '/details');

        return response()->json(['message' => trans('api.record stored successflly')], 200);
    }

    /**
     * Display the specified resource.
     */
    public function getById(Product $product)
    {
        $product = new ProductRescource($product);

        return  response()->json(['product' => $product], 200);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(ProductRequest $request, Product $product)
    {
        $data = $request->validated();
        $product->update($data);
        $this->AdminlogActivity('Product', 'update', '/products/' . $product->id . '/details');

        return response()->json(['message' => trans('api.record updated successflly')], 200);
    }

    /**
     * archive the specified resource from storage.
     */
    public function archive(Product $product)
    {
        $product->delete();
        $this->AdminlogActivity('Product', 'archive');

        return response()->json(['message' => trans('api.record archived successflly')], 200);
    }
    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Product $product)
    {
        $product->forceDelete();
        $this->AdminlogActivity('Product', 'delete');
        return response()->json(['message' => trans('api.record deleted successflly')], 200);
    }


    /**
     * restore the specified resource which soft Deleted.
     */
    public function restore(Product $product)
    {

        $product->restore();
        $this->AdminlogActivity('Product', 'restore');
        return response()->json(['message' => trans('api.record restored successflly')], 200);
    }
}
