<?php

namespace App\Http\Controllers\Api\AdminDashboard\Finances;

use App\Exports\TransactionPaymentGatewayExport;
use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use App\Http\Resources\AdminDashboard\Finances\TransactionPaymentGatewayResource;
use App\Models\ManagerDashboard\Settings\Subscription;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Facades\Excel;

class TransactionPaymentGatewayController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {

        $data = Subscription::with('transactionData')->where('paid_type', 'payment_gateway');
        $paid_bills = Subscription::with('transactionData')->where('paid_type', 'payment_gateway')->where('paid_status', 'paid')->count();
        $cancaled_bills = Subscription::with('transactionData')->where('paid_type', 'payment_gateway')->where('paid_status', 'refused')->count();

        // Retrieve all query parameters
        $queryParams = $request->query();

        $filterData = array_filter($queryParams, function ($key) {
            return $key !== 'per_page';
        }, ARRAY_FILTER_USE_KEY);


        if (!empty($filterData)) {
            $data = Helper::filterData('App\Models\ManagerDashboard\Settings\Subscription', $filterData);
        }
        $data->where('paid_type', 'payment_gateway');

        // Paginate results with a default value if not provided
        $perPage = $queryParams['per_page'] ?? 15;

        // Date Filter
        if (isset($queryParams['date_from_type']) && isset($queryParams['date_from'])) {

            $date_from = $queryParams['date_from'];

            if ($queryParams['date_from_type'] == "hijri") {
                $date_from = Helper::getCoptic($queryParams['date_from']);
            }
            // Parse the Gregorian date from the user input (e.g., 11/12/2024 -> 2024-12-11)
            $date_from = Carbon::createFromFormat('d/m/Y', $date_from)->startOfDay();

            $data = $data->whereHas('transactionPaymentGatewayData', function ($query) use ($date_from) {
                $query->where('trans_date', '>=', $date_from);
            });
        }

        if (isset($queryParams['date_to_type']) && isset($queryParams['date_to'])) {
            $date_to = $queryParams['date_to'];

            if ($queryParams['date_to_type'] == "hijri") {
                // Convert Hijri to Gregorian
                $date_to = Helper::getCoptic($queryParams['date_to']);
            }
            // Parse the Gregorian date from the user input (e.g., 11/12/2024 -> 2024-12-11)
            $date_to = Carbon::createFromFormat('d/m/Y', $date_to)->startOfDay();

            // Apply date_to filter to transactionData relation
            $data = $data->whereHas('transactionPaymentGatewayData', function ($query) use ($date_to) {
                $query->where('trans_date', '<=', $date_to);
            });
        }

        if (isset($request['orderBy']) && $request['orderBy'] === 'asc') {

            $data->orderBy('id', 'asc');
        } elseif (isset($request['orderBy']) && $request['orderBy'] === 'desc') {
            $data->orderBy('id', 'desc');
        }
        ;

        if (isset($queryParams['search'])) {
            $search_text = $queryParams['search'];

            $data->whereHas('user', function ($query) use ($search_text) {
                $query->where('name', 'like', "%$search_text%");
            })
                ->orWhere('invoice_number', 'like', "%$search_text%")
                ->orWhere('order_number', 'like', "%$search_text%");
        }

        $subscriptions = $data->paginate($perPage);

        $subscriptions_resource = TransactionPaymentGatewayResource::collection($subscriptions);

        $response = [
            'total_bills' => $subscriptions->count(),
            'paid_bills' => $paid_bills,
            'cancaled_bills' => $cancaled_bills,
            'subscriptions' => $subscriptions_resource
        ];


        return response()->json(['data' => $response]);
    }


    /**
     * Display the specified resource.
     */
    public function show(Subscription $subscription)
    {
        $transaction_payment_gateway = [
            'id' => $subscription->id,
            'order_number' => $subscription->order_number,
            'invoice_number' => $subscription->invoice_number,
            'total_price' => $subscription->total_price,
            'status' => $subscription->status,
            'paid_status' => $subscription->paid_status,
            'payment_method' => $subscription->transactionPaymentGatewayData?->payment_method,
            'paid_date' => $subscription->transactionData?->transfer_date,
            'city' => $subscription->school->city?->name,
        ];

        $owner_details = [
            'user_id' => $subscription->user->id,
            'name' => $subscription->user->name,
            'email' => $subscription->user->email,
            'phone' => $subscription->user->phone,
        ];

        $company_details = $subscription->school->bussinessData ? [
            'company_name' => $subscription->school->bussinessData?->company_name,
            'sector' => $subscription->school->bussinessData?->sector,
            'tax_registration_number' => $subscription->school->bussinessData?->tax_registration_number,
            'commercial_registration_number' => $subscription->school->bussinessData?->commercial_registration_number,
        ] : [];

        $financial_details = [
            'subscription_fee' => $subscription->subscription_fee,
            'tax_fees' => $subscription->tax_fees,
            'discount_amount' => $subscription->coupon_discount,
            'total_price' => $subscription->total_price,

        ];

        return response()->json([
            'Bank_transfer_details' => $transaction_payment_gateway,
            'owner_details' => $owner_details,
            'company_details' => $company_details,
            'financial_details' => $financial_details
        ]);
    }

    public function export(Request $request)
    {
        $start_date = $request->input('start_date');
        $end_date = $request->input('end_date');
        if (isset($request['start_date_type']) && isset($request['start_date'])) {
            $start_date = $request['start_date'];

            if ($request['start_date_type'] == "hijri") {
                $start_date = Helper::getCoptic($request['start_date']);
            }
        }

        if (isset($request['end_date_type']) && isset($request['end_date'])) {
            $end_date = $request['end_date'];

            if ($request['end_date_type'] == "hijri") {
                $end_date = Helper::getCoptic($request['end_date']);
            }
        }
        return Excel::download(new TransactionPaymentGatewayExport($start_date, $end_date), 'TransactionPaymentGateway.xlsx');
    }
}
