<?php

namespace App\Http\Controllers\Api\AdminDashboard\Finances;

use App\Exports\BankTransactionExport;
use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use App\Http\Resources\AdminDashboard\Finances\BankTransactionResource;
use App\Models\ManagerDashboard\Settings\Subscription;
use App\Traits\SmsTrait;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Facades\Excel;
use Tymon\JWTAuth\Facades\JWTAuth;

class BankTransferController extends Controller
{
    use SmsTrait;
    /**
     * Display a listing of the resource.
     *
     *
     */
    public function index(Request $request)
    {
        $data = Subscription::with('transactionData')->where('paid_type', 'bank_transfer');

        // Retrieve all query parameters
        $queryParams = $request->query();

        $filterData = array_filter($queryParams, function ($key) {
            return $key !== 'per_page';
        }, ARRAY_FILTER_USE_KEY);

        if (!empty($filterData)) {
            $data = Helper::filterData('App\Models\ManagerDashboard\Settings\Subscription', $filterData);
        }

        $data->where('paid_type', 'bank_transfer');

        // Paginate results with a default value if not provided
        $perPage = $queryParams['per_page'] ?? 15;

        // Date Filter
        if (isset($queryParams['date_from_type']) && isset($queryParams['date_from'])) {

            $date_from = $queryParams['date_from'];

            if ($queryParams['date_from_type'] == "hijri") {
                $date_from = Helper::getCoptic($queryParams['date_from']);
            }

            $data = $data->whereHas('transactionData', function ($query) use ($date_from) {
                $query->where('transfer_date', '>=', $date_from);
            });
        }

        if (isset($queryParams['date_to_type']) && isset($queryParams['date_to'])) {
            $date_to = $queryParams['date_to'];

            if ($queryParams['date_to_type'] == "hijri") {
                // Convert Hijri to Gregorian
                $date_to = Helper::getCoptic($queryParams['date_to']);
            }

            // Apply date_to filter to transactionData relation
            $data = $data->whereHas('transactionData', function ($query) use ($date_to) {
                $query->where('transfer_date', '<=', $date_to);
            });
        }

        if (isset($request['orderBy']) && $request['orderBy'] === 'asc') {
            $data->orderBy('id', 'asc');
        } elseif (isset($request['orderBy']) && $request['orderBy'] === 'desc') {
            $data->orderBy('id', 'desc');
        };

        if (isset($queryParams['search'])) {
            $search_text = $queryParams['search'];

            $data->whereHas('school', function ($query) use ($search_text) {
                $query->where('name', 'like', "%$search_text%");
            })->orWhere('invoice_number', 'like', "%$search_text%");
        }

        $subscriptions = $data->paginate($perPage);

        $data_resource = BankTransactionResource::collection($subscriptions);

        return response()->json(['data' => $data_resource]);
    }

    public function confirm(Subscription $subscription)
    {

        $date = Carbon::now();

        $start_date = $date->format('d/m/Y');

        $end_date = $date->copy()->addYear()->format('d/m/Y');


        // Ensure the number is unique or prepend zeros to meet the format if necessary
        $timestamp = time();
        $generateInvoiceNumber = str_pad($timestamp, 10, '0', STR_PAD_LEFT);

        $subscription->update([
            'start_date' => $subscription->service_type == 'system' ? $start_date : null,
            'end_date' => $subscription->service_type == 'system' ? $end_date : null,
            'status' => 'subscribed',
            'paid_status' => 'paid',
            'invoice_number' => $generateInvoiceNumber
        ]);

        $user =  JWTAuth::parseToken()->authenticate();
        $message_text =  " تم تنشيط حسابك فى نظام لام بنجاح.";
        static::sendSMSNotification($subscription->user->phone, $message_text, $user->manager_id);


        return response()->json([
            'message' => 'تم تأكيد الدفع بنجاح',
        ]);
    }

    public function refuse(Request $request, Subscription $subscription)
    {

        $data = $request->validate([
            'reason' => ['required', 'string'],
        ]);

        // Ensure the number is unique or prepend zeros to meet the format if necessary
        $timestamp = time();
        $generateInvoiceNumber = str_pad($timestamp, 10, '0', STR_PAD_LEFT);

        $subscription->update([
            'start_date' => null,
            'end_date' => null,
            'status' => 'not_subscribed',
            'paid_status' => 'refused',
            'invoice_number' => $generateInvoiceNumber,
            'cancellation_reason' => $data['reason'],

        ]);
        $user =  JWTAuth::parseToken()->authenticate();
        $message_text =  " لم يتم قبول التحويل البنكى يرجى مراجعة السبب فى النظام أو التواصل معنا للمساعدة.";
        static::sendSMSNotification($subscription->user->phone, $message_text, $user->manager_id);


        return response()->json([
            'message' => 'تم رفض الطلب بنجاح',
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     */
    public function show(Subscription $subscription)
    {

        $Bank_transfer_details = [
            'id' => $subscription->id,
            'invoice_number' => $subscription->invoice_number,
            'total_price' => $subscription->total_price,
            'paid_status' => $subscription->paid_status,
            'paid_date' => $subscription->transactionData?->transfer_date,
            'transfer_receipt' => $subscription->transactionData?->transfer_receipt ? url('public/uploads/TransferReceipts/' . $subscription->transactionData->transfer_receipt) : Null,
            'city' => $subscription->school->city?->name,
            'created_at' => $subscription->transactionData?->created_at
                ? Carbon::parse($subscription->transactionData->created_at)->format('h:i A d/m/Y')
                : null,
        ];

        $Bank_transfer_owner_details = [
            'user_id' => $subscription->user->id,
            'name' => $subscription->user->name,
            'email' => $subscription->user->email,
            'phone' => $subscription->user->phone,
        ];

        $company_details = $subscription->school->bussinessData ? [
            'company_name' => $subscription->school->bussinessData?->company_name,
            'sector' => $subscription->school->bussinessData?->sector,
            'tax_registration_number' => $subscription->school->bussinessData?->tax_registration_number,
            'commercial_registration_number' => $subscription->school->bussinessData?->commercial_registration_number,
        ] : [];

        $financial_details = [
            'subscription_fee' => $subscription->subscription_fee,
            'tax_fees' => $subscription->tax_fees,
            'discount_amount' => $subscription->coupon_discount,
            'total_price' => $subscription->total_price,

        ];

        return response()->json([
            'Bank_transfer_details' => $Bank_transfer_details,
            'Bank_transfer_owner_details' => $Bank_transfer_owner_details,
            'company_details' => $company_details,
            'financial_details' => $financial_details
        ]);
    }

    public function export(Request $request)
    {
        $start_date = $request->input('start_date');
        $end_date = $request->input('end_date');
        if (isset($request['start_date_type']) && isset($request['start_date'])) {
            $start_date = $request['start_date'];

            if ($request['start_date_type'] == "hijri") {
                $start_date = Helper::getCoptic($request['start_date']);
            }
        }

        if (isset($request['end_date_type']) && isset($request['end_date'])) {
            $end_date = $request['end_date'];

            if ($request['end_date_type'] == "hijri") {
                $end_date = Helper::getCoptic($request['end_date']);
            }
        }


        return Excel::download(new BankTransactionExport($start_date, $end_date), 'BankTransaction.xlsx');
    }
}
