<?php

namespace App\Http\Controllers\Api\AdminDashboard\Event;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use JWTAuth;
use Tymon\JWTAuth\Exceptions\JWTException;

use Illuminate\Http\Request;
use App\Http\Requests\Event\EventStoreRequest;
use App\Models\Event;
use App\Http\Resources\Event\EventResource;
use App\Models\AdminDashboard\School\SemesterWeekDay;
use App\Traits\LogsActivity;
use Carbon\Carbon;

class EventController extends Controller
{

    use LogsActivity;
    /* to uses the class methods
        -you should be authenticated 
        -you should be authorized and have this permission
    */
    public function __construct()
    {
        // $this->middleware(['auth:api', 'jwt.verify','permission:events']);
    }

    /*
        -this method will return all Model recoreds
    */
    public function getAll(Request $request)
    {
        // Start with the base query
        $data = Event::query();

        // Retrieve all query parameters
        $queryParams = $request->query();
        $filterData = $request->all();
        unset($filterData['date_from_type'], $filterData['date_from']);
        // $filterData = array_filter($queryParams, function ($key) {
        //     return $key !== 'date_type';
        // }, ARRAY_FILTER_USE_KEY);
        // Apply filters if any
        if (!empty($filterData)) {
            $data = Helper::filterData('App\Models\Event', $filterData);
        }

        // Paginate results with a default value if not provided
        $perPage = $queryParams['per_page'] ?? 15;

        if (isset($queryParams['date_from_type']) && isset($queryParams['date_from'])) {
            $date_from = $queryParams['date_from'];

            if ($queryParams['date_from_type'] == "hijri") {
                $date_from = Helper::getCoptic($queryParams['date_from']);
            }
            $data = $data->where('date_from', 'like', "%$date_from%");
        }

        $filterData = EventResource::collection($data->paginate($perPage));

        return response()->json([
            'events' => $filterData,
            'message' => trans('api.list of all available records'),
        ]);
    }

    /* this method will return specific Model recored 
        by using ID
    */
    public function getById($id)
    {
        $event = Event::findOrFail($id);
        $event = new EventResource($event);
        return response()->json(['event' => $event, 'message' => trans('api.event details')]);
    }

    /* this method will stord data to DB Model*/
    public function store(EventStoreRequest $request)
    {
        $data = $request->all();

        $data['date_from'] = $data['date_from_type'] === 'hijri' ? Helper::getCoptic($data['date_from']) : $data['date_from'];
        $data['date_to'] = $data['date_to_type'] === 'hijri' ? Helper::getCoptic($data['date_to']) : $data['date_to'];
        unset($data['date_from_type'], $data['date_to_type']);
        $date_from = Carbon::createFromFormat('d/m/Y', $data['date_from']);
        $date_to = Carbon::createFromFormat('d/m/Y', $data['date_to']);
        $event_check = Event::whereRaw(
            "
            (
                STR_TO_DATE(date_from, '%d/%m/%Y') BETWEEN STR_TO_DATE(?, '%d/%m/%Y') AND STR_TO_DATE(?, '%d/%m/%Y')
                OR
                STR_TO_DATE(date_to, '%d/%m/%Y') BETWEEN STR_TO_DATE(?, '%d/%m/%Y') AND STR_TO_DATE(?, '%d/%m/%Y')
                OR
                (STR_TO_DATE(date_from, '%d/%m/%Y') <= STR_TO_DATE(?, '%d/%m/%Y') 
                AND STR_TO_DATE(date_to, '%d/%m/%Y') >= STR_TO_DATE(?, '%d/%m/%Y'))
            )",
            [$date_from->format('d/m/Y'), $date_to->format('d/m/Y'), $date_from->format('d/m/Y'), $date_to->format('d/m/Y'), $date_from->format('d/m/Y'), $date_to->format('d/m/Y')]
        )->count();
        if ($event_check >= 2) {
            return response()->json(['message' => 'لديك بالفعل حدثين لليوم فى هذه المدة الزمنية'], 422);
        }
        $event = Event::create($data);
        $dateFrom = $date_from;
        $dateTo = $date_to;
        while ($dateFrom->lte($dateTo)) {
            $semester = null;

            while (!$semester) {
                $semester = SemesterWeekDay::where('date', $dateFrom->format('d/m/Y'))->first();

                if ($semester) {
                    $data['semester_id'] = $semester->semester_id;
                }
                if (!$semester) {
                    $dateFrom->addDay();
                }
            }

            $event->task()->create([
                'event_id' => $event->id,
                'details' => $event->title,
                'date' => $dateFrom->format('d/m/Y'),
                'appointed' => 0,
                'type' => 'event',
                'semester_id' => $semester->semester_id,
            ]);

            $dateFrom->addDay();
        }

        // Log the activity for the created event
        $this->AdminlogActivity('event', 'store', '/events/' . $event->id);

        return response()->json(['message' => trans('api.record stored successflly')]);
    }

    /* this method will update specific Model recored 
        by using ID
    */
    public function update(EventStoreRequest $request, $id)
    {

        $data = $request->all();
        $data['date_from'] = $data['date_from_type'] === 'hijri' ? Helper::getCoptic($data['date_from']) : $data['date_from'];
        $data['date_to'] = $data['date_to_type'] === 'hijri' ? Helper::getCoptic($data['date_to']) : $data['date_to'];
        unset($data['date_from_type'], $data['date_to_type']);
        $date_from = Carbon::createFromFormat('d/m/Y', $data['date_from']);
        $date_to = Carbon::createFromFormat('d/m/Y', $data['date_to']);
        $event_check = Event::whereRaw(
            "
            (
                STR_TO_DATE(date_from, '%d/%m/%Y') BETWEEN STR_TO_DATE(?, '%d/%m/%Y') AND STR_TO_DATE(?, '%d/%m/%Y')
                OR
                STR_TO_DATE(date_to, '%d/%m/%Y') BETWEEN STR_TO_DATE(?, '%d/%m/%Y') AND STR_TO_DATE(?, '%d/%m/%Y')
                OR
                (STR_TO_DATE(date_from, '%d/%m/%Y') <= STR_TO_DATE(?, '%d/%m/%Y') 
                AND STR_TO_DATE(date_to, '%d/%m/%Y') >= STR_TO_DATE(?, '%d/%m/%Y'))
            )",
            [$date_from->format('d/m/Y'), $date_to->format('d/m/Y'), $date_from->format('d/m/Y'), $date_to->format('d/m/Y'), $date_from->format('d/m/Y'), $date_to->format('d/m/Y')]
        )->where('id', '!=', $id)->count();
        if ($event_check >= 2) {
            return response()->json(['message' => 'لديك بالفعل حدثين لليوم فى هذه المدة الزمنية'], 422);
        }
        $event = Event::findOrFail($id);
        if (!$event->holiday) {
            $event->task()->delete();
            $dateFrom = $date_from;
            $dateTo = $date_to;
            while ($dateFrom->lte($dateTo)) {
                $semester = null;

                while (!$semester) {
                    $semester = SemesterWeekDay::where('date', $dateFrom->format('d/m/Y'))->first();

                    if ($semester) {
                        $data['semester_id'] = $semester->semester_id;
                    }
                    if (!$semester) {
                        $dateFrom->addDay();
                    }
                }

                $event->task()->create([
                    'event_id' => $event->id,
                    'details' => $event->title,
                    'date' => $dateFrom->format('d/m/Y'),
                    'appointed' => 0,
                    'type' => 'event',
                    'semester_id' => $semester->semester_id,
                ]);

                $dateFrom->addDay();
            }
        }
        $event->update($data);

        $this->AdminlogActivity('event', 'update', '/events/' . $id);
        return response()->json(['message' => trans('api.record updated successflly')]);
    }
    public function holiday(Event $event)
    {
        if (!$event->holiday) { // update from not holiday to holiday and delete tasks
            $event->task()->delete();
        } else {
            // update from holiday to not holiday and add tasks
            $dateFrom = Carbon::createFromFormat('d/m/Y', $event->date_from);
            $dateTo = Carbon::createFromFormat('d/m/Y', $event->date_to);
            while ($dateFrom->lte($dateTo)) {
                $semester = null;

                while (!$semester) {
                    $semester = SemesterWeekDay::where('date', $dateFrom->format('d/m/Y'))->first();

                    if ($semester) {
                        $data['semester_id'] = $semester->semester_id;
                    }
                    if (!$semester) {
                        $dateFrom->addDay();
                    }
                }

                $event->task()->create([
                    'event_id' => $event->id,
                    'details' => $event->title,
                    'date' => $dateFrom->format('d/m/Y'),
                    'appointed' => 0,
                    'type' => 'event',
                    'semester_id' => $semester->semester_id,
                ]);

                $dateFrom->addDay();
            }
        }
        $event->update(['holiday' => !$event->holiday]);

        return response()->json(['message' => trans('api.record updated successflly')]);
    }

    /* this method will archive specific Model recored 
        by using ID
    */
    public function archive($id)
    {
        Event::findOrFail($id)->delete();
        $this->AdminlogActivity('event', 'archive');
        return response()->json(['message' => trans('api.record archived successflly')]);
    }

    /* this method will delete specific Model recored 
        by using ID
    */
    public function delete($id)
    {
        Event::findOrFail($id)->forceDelete();
        $this->AdminlogActivity('event', 'delete');
        return response()->json(['message' => trans('api.record deleted successflly')]);
    }

    /* this method will restore specific Model recored 
        by using ID
    */
    public function restore($id)
    {
        Event::withTrashed()->findOrFail($id)->restore();
        $this->AdminlogActivity('event', 'restore');
        return response()->json(['message' => trans('api.record restored successflly')]);
    }
}
