<?php

namespace App\Http\Controllers\Api\ManagerDashboard\Messages;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use App\Http\Requests\ManagerDashboard\Messages\MessageStoreRequest ;
use App\Http\Requests\ManagerDashboard\Messages\MessageUpdateRequest;
use App\Http\Resources\ManagerDashboard\Messages\MessageHistoryResource;
use App\Http\Resources\ManagerDashboard\Messages\MessageResource;
use App\Models\AdminDashboard\School\SchoolEmployee;
use App\Models\AdminDashboard\School\Student;
use App\Models\ManagerDashboard\Messages\MessageReciver;
use App\Models\ManagerDashboard\Messages\Message;
use App\Models\User;
use App\Traits\LogsActivity;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Collection;
use JWTAuth;
use Tymon\JWTAuth\Exceptions\JWTException;

use Illuminate\Http\Request;

class MessagesController extends Controller
{
    use LogsActivity;
    /* to uses the class methods
        -you should be authenticated 
        -you should be authorized and have this permission
    */
    public function __construct()
    {
        // $this->middleware(['auth:api', 'jwt.verify','permission:messgaes']);
    }

    /*
        -this method will return all Model recoreds
    */
    public function getAll(Request $request, $school_id)
    {
        // Initialize the query with the school_id
        $data = Message::where('school_id', $school_id);

        // Retrieve all query parameters
        $queryParams = $request->query();
        $filterData = array_filter($queryParams, function ($key) {
            return $key !== 'name';
        }, ARRAY_FILTER_USE_KEY);

        // Apply filters if any
        if (!empty($filterData)) {
            $data = Helper::filterData('App\Models\ManagerDashboard\Messages\Message', $filterData);
        }

        // Custom filter date
        if (isset($queryParams['date_type']) && isset($queryParams['date'])) {
            $date = $queryParams['date'];
            if ($queryParams['date_type'] == "hijri") {
                $date = Helper::getCoptic($queryParams['date']);
            }
            $date = Carbon::createFromFormat('d/m/Y', $date)->format('Y-m-d');
            $data = $data->whereDate('updated_at', $date);
        }

        // Custom filter by name
        if ($request->name) {
            // Get IDs of students 
            $student_ids = Student::where('name', 'like', '%' . $request->name . '%')
                ->pluck('id')
                ->toArray();
            $student_message_ids = MessageReciver::whereIn('user_id', $student_ids)->where('user_type', 'student')
                ->pluck('message_id')
                ->toArray();

            // Get IDs of employees 
            $employee_ids = SchoolEmployee::where('name', 'like', '%' . $request->name . '%')
                ->pluck('id')
                ->toArray();
            $employee_message_ids = MessageReciver::whereIn('user_id', $employee_ids)->where('user_type', 'employee')
                ->pluck('message_id')
                ->toArray();
            $message_ids = array_merge($student_message_ids, $employee_message_ids);
            $data = $data->whereIn('id', $message_ids);
        }

        $perPage = $queryParams['per_page'] ?? 15;
        $paginatedData = $data->paginate($perPage);

        // Transform the results using MessageResource
        $filterData = MessageResource::collection($paginatedData);

        return response()->json([
            'messages' => $filterData,
            'message' => trans('api.list of all available records'),
        ]);
    }

    /*
        -this method will return all Model recoreds
    */
    public function top_messages($school_id)
    {
        // Fetch student IDs and related notifications
        $student_notifications = MessageReciver::where('user_type', 'student')
            ->whereDate('updated_at', Carbon::today())
            ->take(20)
            ->with('message')
            ->get();

        $employee_notifications = MessageReciver::where('user_type', 'employee')
            ->whereDate('updated_at', Carbon::today())
            ->take(20)
            ->with('message')
            ->get();

        // Combine both notifications
        $all_notifications = $student_notifications->concat($employee_notifications);
        
        // Transform notifications using MessageHistoryResource
        $all_data = MessageHistoryResource::collection($all_notifications);
        
        // Convert to array and filter by school_id
        $filtered_data = $all_data->filter(function ($item) use ($school_id) {
            if($item->message){
                return $item->message['school_id'] == $school_id;
            }
        });
       

        return response()->json([
            'n_consumed' => 200,
            'n_package' => 1000,
            'messages_history' => $filtered_data->values()->toArray(),
        ]);
    }

    /* this method will stord data to DB Model*/
    public function store(MessageStoreRequest $request)
    {
        $message = Message::create([
            'message' => $request->message,
            'type' => 'handy',
            'school_id' => $request->school_id,
        ]);

        foreach ($request->reciver_ids as $reciver_id) {
            MessageReciver::create([
                'message_id' => $message->id,
                'user_id' => $reciver_id,
                'user_type' => $request->user_type
            ]);
        }
        // ------------------------------- School ID -----------------------
        $school_id = $request->school_id;
        $this->ManagerlogActivity('Message', 'store', $school_id, '/manager-messages/top/' . $school_id);

        return response()->json(['message' => trans('api.record stored successflly')]);
    }
    /* this method will stord data to DB Model*/
    public function resend(MessageUpdateRequest $request, $notification_id)
    {
        // Old Messgaes
        $oldMessage = Message::findOrFail($notification_id);
        $user_type = MessageReciver::where('message_id', $notification_id)->first()->user_type;

        // New Messgaes
        $message = Message::create([
            'message' => $oldMessage->message,
            'type' => $oldMessage->type,
            'school_id' => $oldMessage->school_id,
        ]);

        foreach ($request->reciver_ids as $reciver_id) {
            MessageReciver::create([
                'message_id' => $message->id,
                'user_id' => $reciver_id,
                'user_type' => $user_type
            ]);
        }
        // ------------------------------- School ID -----------------------
        $school_id = $oldMessage->school_id;
        $this->ManagerlogActivity('Message', 'resend', $school_id, '/manager-messages/top/' . $school_id);

        return response()->json(['message' => trans('api.record stored successflly')]);
    }

    /* this method will stord data to DB Model*/
    public function sms_store(MessageStoreRequest $request)
    {
        $message = Message::create([
            'message' => $request->message,
            'type' => 'sms'
        ]);

        foreach ($request->reciver_ids as $reciver_id) {
            MessageReciver::create([
                'message_id' => $message->id,
                'user_id' => $reciver_id,
            ]);
        }
        // ------------------------------- School ID -----------------------
        $school_id = $request->school_id;
        $this->ManagerlogActivity('Message', 'store', $school_id, '/manager-messages/top/' . $school_id);
        return response()->json(['message' => trans('api.record stored successflly')]);
    }


    /* this method will delete specific Model recored 
        by using ID
    */
    public function delete($id)
    {
        $notification = Message::findOrFail($id);
        $school_id = $notification->school_id;
        $notification->forceDelete();
        $this->ManagerlogActivity('Message', 'delete', $school_id);

        return response()->json(['message' => trans('api.record deleted successflly')]);
    }

    /* this method will archive specific Model recored 
        by using ID
    */
    public function archive($id)
    {
        $notification = Message::findOrFail($id);
        $school_id = $notification->school_id;
        $notification->delete();
        $this->ManagerlogActivity('Message', 'archive', $school_id);
        return response()->json(['message' => trans('api.record archived successflly')]);
    }
    /* this method will restore specific Model recored 
        by using ID
    */
    public function restore($id)
    {
        $notification = Message::findOrFail($id);
        $school_id = $notification->school_id;
        $notification->restore();
        $this->ManagerlogActivity('Message', 'restore', $school_id);
        return response()->json(['message' => trans('api.record restored successflly')]);
    }
}
