<?php

namespace App\Http\Controllers\Api\AdminDashboard\Event;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use JWTAuth;
use Tymon\JWTAuth\Exceptions\JWTException;

use Illuminate\Http\Request;
use App\Http\Requests\Event\EventStoreRequest;
use App\Http\Requests\Event\EventUpdateRequest;
use App\Models\Event;
use App\Http\Resources\Event\EventResource;
use App\Traits\LogsActivity;

class EventController extends Controller
{

    use LogsActivity;
    /* to uses the class methods
        -you should be authenticated 
        -you should be authorized and have this permission
    */
    public function __construct()
    {
        // $this->middleware(['auth:api', 'jwt.verify','permission:events']);
    }

    /*
        -this method will return all Model recoreds
    */
    public function getAll(Request $request)
    {
        // Start with the base query
        $data = Event::query();

        // Retrieve all query parameters
        $queryParams = $request->query();
        $filterData = array_filter($queryParams, function ($key) {
            return $key !== 'date';
        }, ARRAY_FILTER_USE_KEY);

        // Apply filters if any
        if (!empty($filterData)) {
            $data = Helper::filterData('App\Models\Event', $filterData);
        }

        // Paginate results with a default value if not provided
        $perPage = $queryParams['per_page'] ?? 15;

        if (isset($queryParams['date_type']) && isset($queryParams['date'])) {
            $date = $queryParams['date'];

            if ($queryParams['date_type'] == "hijri") {
                $date = Helper::getCoptic($queryParams['date']);
            }
            $data = $data->where('date', 'like', "%$date%");
        }

        $filterData = EventResource::collection($data->paginate($perPage));

        return response()->json([
            'events' => $filterData,
            'message' => trans('api.list of all available records'),
        ]);
    }

    /* this method will return specific Model recored 
        by using ID
    */
    public function getById($id)
    {
        $event = Event::findOrFail($id);
        $event = new EventResource($event);
        return response()->json(['event' => $event, 'message' => trans('api.event details')]);
    }

    /* this method will stord data to DB Model*/
    public function store(EventStoreRequest $request)
    {
        $data = $request->all();
        if ($data['date_type'] === 'hijri') {
            // Convert the Hijri date to Coptic
            $data['date'] = Helper::getCoptic($data['date']);

            // Optional: Change the date_type to 'coptic' if you want to store it as such
            $data['date_type'] = 'coptic';
        }
        $event = Event::create($data);

        // Log the activity for the created event
        $this->AdminlogActivity('event', 'store', '/events/' . $event->id);


        return response()->json(['message' => trans('api.record stored successflly')]);
    }

    /* this method will update specific Model recored 
        by using ID
    */
    public function update(EventUpdateRequest $request, $id)
    {

        $data = $request->all();
        if ($request->date_type) {
            if ($data['date_type'] === 'hijri') {
                $data['date'] = Helper::getCoptic($data['date']);
                $data['date_type'] = 'coptic';
            }
        }
        Event::findOrFail($id)->update($data);
        $this->AdminlogActivity('event', 'update', '/events/' . $id);
        return response()->json(['message' => trans('api.record updated successflly')]);
    }

    /* this method will archive specific Model recored 
        by using ID
    */
    public function archive($id)
    {
        Event::findOrFail($id)->delete();
        $this->AdminlogActivity('event', 'archive');
        return response()->json(['message' => trans('api.record archived successflly')]);
    }

    /* this method will delete specific Model recored 
        by using ID
    */
    public function delete($id)
    {
        Event::findOrFail($id)->forceDelete();
        $this->AdminlogActivity('event', 'delete');
        return response()->json(['message' => trans('api.record deleted successflly')]);
    }

    /* this method will restore specific Model recored 
        by using ID
    */
    public function restore($id)
    {
        Event::withTrashed()->findOrFail($id)->restore();
        $this->AdminlogActivity('event', 'restore');
        return response()->json(['message' => trans('api.record restored successflly')]);
    }
}
