<?php

namespace App\Http\Controllers\Api\AdminDashboard\Category;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use App\Http\Requests\AdminDashboard\Categories\StoreCategoryRequest;
use App\Http\Requests\AdminDashboard\Categories\UpdateCategoryRequest;
use App\Http\Requests\UpdateStatusRequest;
use App\Http\Resources\AdminDashboard\Category\CategoryResource;
use App\Models\AdminDashboard\Category\Category;
use App\Traits\LogsActivity;
use Illuminate\Http\Request;

class CategoryController extends Controller
{
    use LogsActivity;
    public function __construct()
    {
        //$this->middleware(['auth:api','permission:categories']);
    }

    /**
     * Display a listing of the resource.
     */
    public function getAll(Request $request){
        // Start with the base query
        $data = Category::with('subCategories')->where('parent_id',null);

        // Retrieve all query parameters
        $queryParams = $request->query();

    
        // Apply filters if any
        if (!empty($queryParams)) {
            $data = Helper::filterData('App\Models\AdminDashboard\Category\Category', $queryParams,['subCategories'])->where('parent_id',null);
        }

        // Paginate results with a default value if not provided
        $perPage = $queryParams['per_page'] ?? 15;

        $filterData = CategoryResource::collection($data->paginate($perPage));

        return response()->json([
            'categories' => $filterData,
            'message' => trans('api.list of all available records'),
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreCategoryRequest $request)
    {

        $validatedData = $request->validated();

        $category = Category::create($validatedData);
        $this->AdminlogActivity('Category', 'store', '/categories/' . $category->id.'/details/');

        return response()->json(['message' => trans('api.record stored successflly')], 200);
    }

    /**
     * Display the specified resource.
     */
    public function getById(Category $category)
    {
        $category = new CategoryResource($category->load('subCategories'));

        return response()->json(['category' => $category], 200);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateCategoryRequest $request, Category $category)
    {

        $validatedData = $request->validated();

        $category->update($validatedData);
        $this->AdminlogActivity('Category', 'update', '/categories/' . $category->id . '/details/');


        return response()->json(['message' => trans('api.record updated successflly')], 200);
    }

    /* this method will change activation status for
       specific Model recored by using ID
    */
    public function changeStatus(UpdateStatusRequest $request, Category $category)
    {

        $category->status = $request->status;

        $category->save();

        return response()->json(['message' => trans('api.record updated successflly')], 200);
    }

    /* this method will archive specific Model recored 
        by using ID
    */
    public function archive(Category $category)
    {

        $category->delete();

        $category->subCategories()->delete();
        $this->AdminlogActivity('Category', 'archive');


        return response()->json(['message' => trans('api.record archived successflly')], 200);
    }


    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Category $category)
    {
        $category->forceDelete();
        $this->AdminlogActivity('Category', 'delete');


        return response()->json(['message' => trans('api.record deleted successflly')], 200);
    }

    /**
     * restore the specified resource which soft Deleted.
     */
    public function restore(Category $category)
    {

        $category->restore();

        $category->subCategories()->restore();
        $this->AdminlogActivity('Category', 'restore');


        return response()->json(['message' => trans('api.record restored successflly')], 200);
    }
}
