<?php

namespace App\Http\Controllers\Branch;

use App\Http\Controllers\Controller;
use App\Models\Branch\Appointment;
use App\Models\Branch\Branch;
use App\Models\Branch\Complaint;
use App\Models\Invoice\Debtor;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Validation\Rule;

class ComplaintController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $complaints = Complaint::with(['appointment' => function ($q) {
                    $q->select('id', 'branch_id', 'start_at', 'patient_id', 'doctor_id')
                        ->with(['patient' => function ($q) {
                            $q->select('id', 'avatar', DB::raw('CONCAT(first_Name, " ", second_Name) AS name'))
                        ;}])
                        ->with(['doctor' => function ($q) {
                            $q->select('id', 'avatar', DB::raw('CONCAT(first_Name, " ", second_Name) AS name'))
                        ;}])
                        ->with(['branch' => function ($q) {
                            $q->select('id', 'name');
                        ;}])
                    ;}])->get();

        return view('branch/complaint.index', compact('complaints'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $appointment = Appointment::select('id', 'patient_id', 'doctor_id', 'start_at', 'end_at')
        ->Where('branch_id', Auth::user()->branch_id)
        ->whereDate('start_at', date('Y-m-d'))
        ->with(['patient' => function ($q) {
            $q->select('id', 'avatar', DB::raw('CONCAT(first_Name, " ", second_Name) AS name'));}])
        ->orderBy('id', 'DESC')
        ->get();

        return $appointment;
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $this->validate($request, [
            'first_name' => 'required',
            'second_name' => 'required',
            'avatar' => ['image', 'mimes:jpeg,jpg,png', 'max:200'],
        ]);

        //insert img
        if($request->hasFile('avatar')){
            $file_extension = request()->avatar->getClientOriginalExtension();
            $file_name = 'deb' . $request->input('first_name') . time() . '.' . $file_extension;
            $path = 'img/useravatar';
            $request->avatar->move($path, $file_name);
        }
        else{
            $file_name = 'default-pp.png';
        };
        
        $debtor = Debtor::create([  
            'first_name' => $request->input('first_name'),
            'second_name' => $request->input('second_name'),
            'avatar' => $file_name,
            'company_name' => $request->input('company_name'),
        ]);

        return redirect()->route('sett.debtorcat.index')
            ->with('success', 'Debtor has created successfully');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $complaints = Complaint::with(['appointment' => function ($q) {
            $q->select('id', 'branch_id', 'start_at', 'patient_id', 'doctor_id', 'creator_id')
                ->with(['branch' => function ($q) {
                    $q->select('id', 'name');
                ;}])
                ->with(['patient' => function ($q) {
                    $q->select('id', 'avatar', DB::raw('CONCAT(first_Name, " ", second_Name) AS name'), 'phone_number')
                ;}])
                ->with(['doctor' => function ($q) {
                    $q->select('id', 'avatar', DB::raw('CONCAT(first_Name, " ", second_Name) AS name'))
                ;}])
                ->with(['creator' => function ($q) {
                    $q->select('id', 'avatar', DB::raw('CONCAT(first_Name, " ", second_Name) AS name'))
                ;}])
        ;}])->find($id);

        return view('branch/complaint.show', compact('complaints'));
    }


    public function complaint_show_ajax()
    {
        $appointment = Appointment::select('id', 'patient_id', 'doctor_id', 'start_at', 'end_at', 'rate')
        ->Where('branch_id', Auth::user()->branch_id)
        ->whereDate('start_at', date('Y-m-d'))
        ->whereIn('status', [1,2,3,4])
        ->whereNull('rate')
        ->with(['patient' => function ($q) {
            $q->select('id', 'avatar', DB::raw('CONCAT(first_Name, " ", second_Name) AS name'));}])
        ->orderBy('id', 'DESC')
        ->get();

        return $appointment;
    }


    public function complaint_save_ajax(Request $request){
        
        $complaint = Complaint::create([  
            'appointment_id' => $request->input('appointment_id'),
            'body' => $request->input('body'),
        ]);

        if ($complaint){
            return 'Thanks! for your feedback';
        }else{
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
    
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $this->validate($request, [
            'type' => ['required', Rule::in(['0', '1'])],
            'answer' => 'required',
        ]);

        $complaint = Complaint::find($id);
        $complaint->status = $request->input('type');
        $complaint->answered = $request->input('answer');
        $complaint->save();


        session()->flash('success', 'Complaint has updated successfully');
        return redirect()->back();
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
    }
}